/**
 ********************************************************************************
 * Copyright (c) 2018-2020 Robert Bosch GmbH and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Robert Bosch GmbH - initial API and implementation
 ********************************************************************************
 */

package org.eclipse.app4mc.amalthea.converters091.impl;

import java.io.File;
import java.util.List;
import java.util.Map;

import org.eclipse.app4mc.amalthea.converters.common.ServiceConstants;
import org.eclipse.app4mc.amalthea.converters.common.base.ICache;
import org.eclipse.app4mc.amalthea.converters.common.base.IConverter;
import org.eclipse.app4mc.amalthea.converters.common.converter.AbstractConverter;
import org.eclipse.app4mc.amalthea.converters.common.utils.AmaltheaNamespaceRegistry;
import org.eclipse.app4mc.amalthea.converters.common.utils.HelperUtil;
import org.jdom2.Attribute;
import org.jdom2.Document;
import org.jdom2.Element;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class is responsible for converting the CustomProperties elements from 0.9.0 to 0.9.1 version format of AMALTHEA model
 *
 * @author zmeer
 *
 */
@Component(
		property = {
			ServiceConstants.INPUT_MODEL_VERSION_PROPERTY + "=0.9.0",
			ServiceConstants.OUTPUT_MODEL_VERSION_PROPERTY + "=0.9.1"},
		service = IConverter.class)

public class CustomPropertiesConverter extends AbstractConverter {

	private static final Logger LOGGER = LoggerFactory.getLogger(CustomPropertiesConverter.class);

	@Override
	@Activate
	protected void activate(Map<String, Object> properties) {
		super.activate(properties);
	}

	@Override
	public void convert(File targetFile, Map<File, Document> filename2documentMap, List<ICache> caches) {

		LOGGER.info("Migration from 0.9.0 to 0.9.1 : Executing CustomProperties converter for model file : {}",
				targetFile.getName());

		final Document root = filename2documentMap.get(targetFile);

		if (root == null) {
			return;
		}
		final Element rootElement = root.getRootElement();

		migrateTimeObject(rootElement);
	}

	private void migrateTimeObject(Element rootElement) {

		final StringBuilder xpathBuffer = new StringBuilder();

		xpathBuffer.append(".//value[@xsi:type=\"am:TimeObject\"]");
		xpathBuffer.append("|");
		xpathBuffer.append(".//values[@xsi:type=\"am:TimeObject\"]");

		final List<Element> timeObjectElements = HelperUtil.getXpathResult(
				rootElement,
				xpathBuffer.toString(),
				Element.class,
				AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));

		for (Element timeObjectElement : timeObjectElements) {
			Attribute typeAttribute = timeObjectElement.getAttribute("type", AmaltheaNamespaceRegistry.getGenericNamespace("xsi"));
			if(typeAttribute!=null) {
				typeAttribute.setValue("am:Time");
			}
		}
	}

}