/*******************************************************************************
 * Copyright (c) 2008, 2025 IBM Corporation and others.
 *
 * This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.pde.internal.ui.correction.java;

import java.text.MessageFormat;
import java.util.Arrays;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Status;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IOpenable;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.rewrite.ImportRewrite;
import org.eclipse.jdt.ui.text.java.ClasspathFixProcessor.ClasspathFixProposal;
import org.eclipse.jdt.ui.text.java.IJavaCompletionProposal;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.contentassist.IContextInformation;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;
import org.eclipse.ltk.core.refactoring.TextFileChange;
import org.eclipse.osgi.service.resolver.BundleDescription;
import org.eclipse.osgi.service.resolver.ExportPackageDescription;
import org.eclipse.osgi.util.NLS;
import org.eclipse.pde.core.IBaseModel;
import org.eclipse.pde.core.plugin.IPluginImport;
import org.eclipse.pde.core.plugin.IPluginModelBase;
import org.eclipse.pde.internal.core.ICoreConstants;
import org.eclipse.pde.internal.core.bundle.BundlePluginBase;
import org.eclipse.pde.internal.core.ibundle.IBundle;
import org.eclipse.pde.internal.core.ibundle.IBundlePluginModelBase;
import org.eclipse.pde.internal.core.ibundle.IManifestHeader;
import org.eclipse.pde.internal.core.project.PDEProject;
import org.eclipse.pde.internal.core.text.bundle.ExportPackageHeader;
import org.eclipse.pde.internal.core.text.bundle.ExportPackageObject;
import org.eclipse.pde.internal.core.text.bundle.ImportPackageHeader;
import org.eclipse.pde.internal.core.text.bundle.ImportPackageObject;
import org.eclipse.pde.internal.core.util.ManifestUtils;
import org.eclipse.pde.internal.ui.PDEPlugin;
import org.eclipse.pde.internal.ui.PDEPluginImages;
import org.eclipse.pde.internal.ui.PDEUIMessages;
import org.eclipse.pde.internal.ui.util.ModelModification;
import org.eclipse.pde.internal.ui.util.PDEModelUtility;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.text.edits.TextEdit;
import org.osgi.framework.Constants;
import org.osgi.framework.Version;
import org.osgi.framework.VersionRange;

/**
 * A factory class used to create resolutions for JDT problem markers which involve modifying a project's MANIFEST.MF (or possibly plugin.xml)
 * @since 3.4
 */
public class JavaResolutionFactory {

	/**
	 * Type constant for a proposal of type IJavaCompletionProposal
	 */
	public static final int TYPE_JAVA_COMPLETION = 0x01;
	/**
	 * Type constant for a proposal of type ClasspathFixProposal
	 */
	public static final int TYPE_CLASSPATH_FIX = 0x02;

	/**
	 * This class represents a Change which will be applied to a Manifest file.  This change is meant to be
	 * used to create an IJavaCompletionProposal or ClasspathFixProposal.
	 */
	static abstract class AbstractManifestChange<T> extends Change {

		private final T fChangeObject;
		private final IProject fProject;
		private final CompilationUnit fCompilationUnit;
		private final String fQualifiedTypeToImport;

		public AbstractManifestChange(IProject project, T changeObj, CompilationUnit cu, String qualifiedTypeToImport) {
			fProject = project;
			fChangeObject = changeObj;
			fCompilationUnit = cu;
			fQualifiedTypeToImport = qualifiedTypeToImport;
		}

		protected T getChangeObject() {
			return fChangeObject;
		}

		protected IProject getProject() {
			return fProject;
		}

		protected CompilationUnit getCompilationUnit() {
			return fCompilationUnit;
		}

		protected String getQualifiedTypeToImport() {
			return fQualifiedTypeToImport;
		}

		/*
		 * Provides an image for the Change
		 */
		public abstract Image getImage();

		/*
		 * Provides a description for the Change
		 */
		public abstract String getDescription();

		/*
		 * Added to allow creation of an "undo" change for each AbstractManifestChange
		 */
		protected boolean isUndo() {
			return false;
		}

		protected void insertImport(CompilationUnit compilationUnit, String qualifiedTypeToImport, IProgressMonitor pm)
				throws CoreException {
			if (compilationUnit == null || qualifiedTypeToImport == null) {
				return;
			}
			ImportRewrite rewrite = ImportRewrite.create(compilationUnit, true);
			if (rewrite == null) {
				return;
			}
			if (!isUndo()) {
				rewrite.addImport(qualifiedTypeToImport);
			} else {
				rewrite.removeImport(qualifiedTypeToImport);
			}
			TextEdit rewriteImports = rewrite.rewriteImports(pm);
			ICompilationUnit iCompilationUnit = (ICompilationUnit) compilationUnit.getJavaElement()
					.getAdapter(IOpenable.class);
			performTextEdit(rewriteImports, (IFile) iCompilationUnit.getResource(), pm);
		}

		private void performTextEdit(TextEdit textEdit, IFile file, IProgressMonitor pm)
				throws CoreException {
			TextFileChange textFileChange = new TextFileChange("Add import for " + fQualifiedTypeToImport, file); //$NON-NLS-1$
			textFileChange.setSaveMode(TextFileChange.KEEP_SAVE_STATE);
			textFileChange.setEdit(textEdit);
			textFileChange.perform(pm);
		}

		@Override
		public RefactoringStatus isValid(IProgressMonitor pm) throws CoreException, OperationCanceledException {
			return RefactoringStatus.create(Status.OK_STATUS);
		}

		@Override
		public Object getModifiedElement() {
			return getProject();
		}

		@Override
		public void initializeValidationData(IProgressMonitor pm) {
		}
	}

	/*
	 * A Change which will add a Require-Bundle entry to resolve the given
	 * dependency or add multiple Require-Bundle entries to resolve the dependency
	 * based on description name
	 */
	private static class RequireBundleManifestChange extends AbstractManifestChange<BundleDescription> {

		RequireBundleManifestChange(IProject project, BundleDescription desc, CompilationUnit cu,
				String qualifiedTypeToImport) {
			super(project, desc, cu, qualifiedTypeToImport);
		}

		@Override
		public Change perform(IProgressMonitor pm) throws CoreException {
			PDEModelUtility.modifyModel(new ModelModification(getProject()) {
				@Override
				protected void modifyModel(IBaseModel model, IProgressMonitor monitor) throws CoreException {
					if (!(model instanceof IPluginModelBase base)) {
						return;
					}
					BundleDescription requiredBundle = getChangeObject();
					String pluginId = requiredBundle.getSymbolicName();
					VersionRange versionRange = ManifestUtils
							.createConsumerRequirementRange(requiredBundle.getVersion()).orElse(null);
					IPluginImport[] imports = base.getPluginBase().getImports();
					if (!isUndo()) {
						if (Arrays.stream(imports).map(IPluginImport::getId).anyMatch(pluginId::equals)) {
							return;
						}
						IPluginImport impt = base.getPluginFactory().createImport();
						impt.setId(pluginId);
						if (versionRange != null) {
							impt.setVersion(versionRange.toString());
						}
						base.getPluginBase().add(impt);
					} else {
						for (IPluginImport pluginImport : imports) {
							if (pluginImport.getId().equals(pluginId) && (versionRange == null
									|| versionRange.includes(Version.parseVersion(pluginImport.getVersion())))) {
								base.getPluginBase().remove(pluginImport);
							}
						}
					}
				}
			}, new NullProgressMonitor());

			insertImport(getCompilationUnit(), getQualifiedTypeToImport(), pm);

			if (!isUndo()) {
				return new RequireBundleManifestChange(getProject(), getChangeObject(), getCompilationUnit(),
						getQualifiedTypeToImport()) {
					@Override
					public boolean isUndo() {
						return true;
					}
				};
			}
			return null;
		}


		@Override
		public Image getImage() {
			return PDEPlugin.getDefault().getLabelProvider().get(PDEPluginImages.DESC_REQ_PLUGIN_OBJ);
		}

		@Override
		public String getDescription() {
			return PDEUIMessages.UnresolvedImportFixProcessor_2;
		}

		@Override
		public String getName() {
			BundleDescription requiredBundle = getChangeObject();
			return MessageFormat.format(
					!isUndo() ? PDEUIMessages.UnresolvedImportFixProcessor_0
							: PDEUIMessages.UnresolvedImportFixProcessor_1,
					requiredBundle.getName(), getRequirementVersion(requiredBundle.getVersion()));
		}

		@Override
		public Object getModifiedElement() {
			IFile[] files = new IFile[] {PDEProject.getManifest(getProject()), PDEProject.getPluginXml(getProject())};
			for (IFile file : files) {
				if (file.exists()) {
					return file;
				}
			}
			return super.getModifiedElement();
		}
	}

	/*
	 * A Change which will add an Import-Package entry to resolve the given dependency
	 */
	private static class ImportPackageManifestChange extends AbstractManifestChange<ExportPackageDescription> {

		ImportPackageManifestChange(IProject project, ExportPackageDescription desc, CompilationUnit cu,
				String qualifiedTypeToImport) {
			super(project, desc, cu, qualifiedTypeToImport);
		}

		@Override
		public Change perform(IProgressMonitor pm) throws CoreException {
			PDEModelUtility.modifyModel(new ModelModification(getProject()) {
				@Override
				protected void modifyModel(IBaseModel model, IProgressMonitor monitor) throws CoreException {
					if (!(model instanceof IBundlePluginModelBase base)) {
						return;
					}
					IBundle bundle = base.getBundleModel().getBundle();
					ExportPackageDescription desc = getChangeObject();
					String pkgId = desc.getName();
					IManifestHeader header = bundle.getManifestHeader(Constants.IMPORT_PACKAGE);
					if (header == null) {
						header = bundle.getModel().getFactory().createHeader(Constants.IMPORT_PACKAGE, pkgId);
					}
					if (header instanceof ImportPackageHeader ipHeader) {
						String versionAttr = (BundlePluginBase.getBundleManifestVersion(bundle) < 2)
								? ICoreConstants.PACKAGE_SPECIFICATION_VERSION
								: Constants.VERSION_ATTRIBUTE;
						ImportPackageObject impObject = new ImportPackageObject(ipHeader, desc, versionAttr);
						if (!isUndo()) {
							ipHeader.addPackage(impObject);
						} else {
							ipHeader.removePackage(impObject);
						}
					}
				}
			}, new NullProgressMonitor());

			insertImport(getCompilationUnit(), getQualifiedTypeToImport(), pm);

			if (!isUndo()) {
				return new ImportPackageManifestChange(getProject(), getChangeObject(), getCompilationUnit(),
						getQualifiedTypeToImport()) {
					@Override
					public boolean isUndo() {
						return true;
					}
				};
			}
			return null;
		}

		@Override
		public String getDescription() {
			return PDEUIMessages.UnresolvedImportFixProcessor_5;
		}

		@Override
		public Image getImage() {
			return PDEPlugin.getDefault().getLabelProvider().get(PDEPluginImages.DESC_BUNDLE_OBJ);
		}

		@Override
		public String getName() {
			ExportPackageDescription importedPackage = getChangeObject();
			return MessageFormat.format(
					!isUndo() ? PDEUIMessages.UnresolvedImportFixProcessor_3
							: PDEUIMessages.UnresolvedImportFixProcessor_4,
					importedPackage.getName(), getRequirementVersion(importedPackage.getVersion()));
		}

		@Override
		public Object getModifiedElement() {
			IFile file = PDEProject.getManifest(getProject());
			if (file.exists()) {
				return file;
			}
			return super.getModifiedElement();
		}
	}

	private static Version getRequirementVersion(Version bundleVersion) {
		return new Version(bundleVersion.getMajor(), bundleVersion.getMinor(), 0);
	}

	private static class ExportPackageChange extends AbstractManifestChange<IPackageFragment> {

		ExportPackageChange(IProject project, IPackageFragment fragment) {
			super(project, fragment, null, null);
		}

		@Override
		public Change perform(IProgressMonitor pm) throws CoreException {
			ModelModification mod = new ModelModification(getProject()) {
				@Override
				protected void modifyModel(IBaseModel model, IProgressMonitor monitor) throws CoreException {
					if (model instanceof IBundlePluginModelBase base) {
						IBundle bundle = base.getBundleModel().getBundle();

						ExportPackageHeader header = (ExportPackageHeader) bundle
								.getManifestHeader(Constants.EXPORT_PACKAGE);
						if (header == null) {
							bundle.setHeader(Constants.EXPORT_PACKAGE, ""); //$NON-NLS-1$
							header = (ExportPackageHeader) bundle.getManifestHeader(Constants.EXPORT_PACKAGE);
						}
						header.addPackage(
								new ExportPackageObject(header, getChangeObject(), Constants.VERSION_ATTRIBUTE));
					}
				}
			};
			PDEModelUtility.modifyModel(mod, new NullProgressMonitor());
			// No plans to use as ClasspathFixProposal, therefore we don't have
			// to worry about an undo
			return null;
		}

		@Override
		public String getName() {
			return NLS.bind(PDEUIMessages.ForbiddenAccessProposal_quickfixMessage,
					new String[] { getChangeObject().getElementName(), getProject().getName() });
		}

		@Override
		public Image getImage() {
			return PDEPluginImages.get(PDEPluginImages.OBJ_DESC_BUNDLE);
		}

		@Override
		public Object getModifiedElement() {
			IFile file = PDEProject.getManifest(getProject());
			if (file.exists()) {
				return file;
			}
			return super.getModifiedElement();
		}

		@Override
		public String getDescription() {
			// No plans to use as ClasspathFixProposal, therefore we don't have
			// to implement a description
			return null;
		}
	}

	/**
	 * Creates and returns a proposal which create a Require-Bundle entry in the
	 * MANIFEST.MF (or corresponding plugin.xml) for the supplier of desc. The
	 * object will be of the type specified by the type argument.
	 *
	 * @param project
	 *            the project to be updated
	 * @param desc
	 *            an ExportPackageDescription from the bundle that is to be
	 *            added as a Require-Bundle dependency
	 * @param qualifiedTypeToImport
	 *            the qualified type name of the type that requires this
	 *            proposal. If this argument and cu are supplied the proposal
	 *            will add an import statement for this type to the source file
	 *            in which the proposal was invoked.
	 * @param cu
	 *            the AST root of the java source file in which this fix was
	 *            invoked
	 */
	public static AbstractManifestChange<BundleDescription> createRequireBundleChange(IProject project,
			BundleDescription desc, CompilationUnit cu, String qualifiedTypeToImport) {
		return new RequireBundleManifestChange(project, desc, cu, qualifiedTypeToImport);
	}

	/**
	 * Creates and returns a proposal which create an Import-Package entry in
	 * the MANIFEST.MF for the package represented by desc. The object will be
	 * of the type specified by the type argument.
	 *
	 * @param project
	 *            the project to be updated
	 * @param desc
	 *            an ExportPackageDescription which represents the package to be
	 *            added
	 * @param qualifiedTypeToImport
	 *            the qualified type name of the type that requires this
	 *            proposal. If this argument and cu are supplied the proposal
	 *            will add an import statement for this type to the source file
	 *            in which the proposal was invoked.
	 * @param cu
	 *            the AST root of the java source file in which this fix was
	 *            invoked
	 */
	public static AbstractManifestChange<ExportPackageDescription> createImportPackageChange(IProject project,
			ExportPackageDescription desc, CompilationUnit cu, String qualifiedTypeToImport) {
		return new ImportPackageManifestChange(project, desc, cu, qualifiedTypeToImport);
	}

	public static final IJavaCompletionProposal createSearchRepositoriesProposal(String packageName) {
		return new SearchRepositoriesForIUProposal(packageName);
	}

	/**
	 * Creates and returns a proposal which create an Export-Package entry in the MANIFEST.MF for the package represented by
	 * pkg.  The object will be of the type specified by the type argument.
	 * @param project the project to be updated
	 * @param pkg an IPackageFragment which represents the package to be added
	 */
	public static AbstractManifestChange<IPackageFragment> createExportPackageChange(IProject project,
			IPackageFragment pkg) {
		return new ExportPackageChange(project, pkg);
	}

	// Methods to wrap a AbstractMethodChange into a consumable format

	/**
	 * Creates and returns a ClasspathFixProposal for the given AbstractManifestChange
	 * @param change the modification which should be performed by the proposal
	 * @since 3.4
	 * @see AbstractManifestChange
	 */
	public final static ClasspathFixProposal createClasspathFixProposal(AbstractManifestChange<?> change,
			int relevance) {
		return new ClasspathFixProposal() {

			@Override
			public Change createChange(IProgressMonitor monitor) throws CoreException {
				return change;
			}

			@Override
			public String getAdditionalProposalInfo() {
				return change.getDescription();
			}

			@Override
			public String getDisplayString() {
				return change.getName();
			}

			@Override
			public Image getImage() {
				return change.getImage();
			}

			@Override
			public int getRelevance() {
				return relevance;
			}

		};
	}

	/**
	 * Creates and returns an IJavaCompletionProposal for the given AbstractManifestChange with the given relevance.
	 * @param change the modification which should be performed by the proposal
	 * @param relevance the relevance of the IJavaCompletionProposal
	 * @since 3.4
	 * @see AbstractManifestChange
	 */
	public final static IJavaCompletionProposal createJavaCompletionProposal(AbstractManifestChange<?> change,
			int relevance) {
		return new IJavaCompletionProposal() {

			@Override
			public int getRelevance() {
				return relevance;
			}

			@Override
			public void apply(IDocument document) {
				try {
					change.perform(new NullProgressMonitor());
				} catch (CoreException e) {
				}
			}

			@Override
			public String getAdditionalProposalInfo() {
				return change.getDescription();
			}

			@Override
			public IContextInformation getContextInformation() {
				return null;
			}

			@Override
			public String getDisplayString() {
				return change.getName();
			}

			@Override
			public Image getImage() {
				return change.getImage();
			}

			@Override
			public Point getSelection(IDocument document) {
				return null;
			}
		};
	}

}
