/**
 * Copyright (c) 2016 Kiel University and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Kiel University - initial API and implementation
 */
package org.eclipse.elk.core.meta.jvmmodel;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.LinkedList;
import java.util.List;
import org.eclipse.elk.core.meta.metaData.MdAlgorithm;
import org.eclipse.elk.core.meta.metaData.MdBundle;
import org.eclipse.elk.core.meta.metaData.MdBundleMember;
import org.eclipse.elk.core.meta.metaData.MdCategory;
import org.eclipse.elk.core.meta.metaData.MdGraphFeature;
import org.eclipse.elk.core.meta.metaData.MdGroup;
import org.eclipse.elk.core.meta.metaData.MdGroupOrOption;
import org.eclipse.elk.core.meta.metaData.MdModel;
import org.eclipse.elk.core.meta.metaData.MdOption;
import org.eclipse.elk.core.meta.metaData.MdOptionDependency;
import org.eclipse.elk.core.meta.metaData.MdOptionSupport;
import org.eclipse.elk.core.meta.metaData.MdOptionTargetType;
import org.eclipse.elk.graph.properties.IProperty;
import org.eclipse.elk.graph.properties.Property;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtend2.lib.StringConcatenationClient;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmEnumerationType;
import org.eclipse.xtext.common.types.JvmField;
import org.eclipse.xtext.common.types.JvmFormalParameter;
import org.eclipse.xtext.common.types.JvmGenericType;
import org.eclipse.xtext.common.types.JvmMember;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.common.types.JvmParameterizedTypeReference;
import org.eclipse.xtext.common.types.JvmPrimitiveType;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.common.types.JvmVisibility;
import org.eclipse.xtext.common.types.util.Primitives;
import org.eclipse.xtext.util.Strings;
import org.eclipse.xtext.xbase.XExpression;
import org.eclipse.xtext.xbase.jvmmodel.AbstractModelInferrer;
import org.eclipse.xtext.xbase.jvmmodel.IJvmDeclaredTypeAcceptor;
import org.eclipse.xtext.xbase.jvmmodel.JvmTypesBuilder;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * Infers a JVM model from the source model.
 * 
 * <p>The JVM model should contain all elements that would appear in the Java code which is generated from
 * the source model. Other models link against the JVM model rather than the source model.</p>
 */
@SuppressWarnings("all")
public class MetaDataJvmModelInferrer extends AbstractModelInferrer {
  /**
   * Convenience API to build and initialize JVM types and their members.
   */
  @Inject
  @Extension
  private JvmTypesBuilder _jvmTypesBuilder;
  
  /**
   * Convenience API to wrap primitives.
   */
  @Inject
  @Extension
  private Primitives _primitives;
  
  /**
   * The dispatch method {@code infer} is called for each instance of the
   * given element's type that is contained in a resource.
   * 
   * @param model
   *            the model to create one or more {@link JvmDeclaredType declared types} from.
   * @param acceptor
   *            each created {@link JvmDeclaredType type} without a container should be passed to the acceptor
   *            in order to get attached to the current resource. The acceptor's
   *            {@link IJvmDeclaredTypeAcceptor#accept(org.eclipse.xtext.common.types.JvmDeclaredType) accept(..)}
   *            method takes the constructed empty type for the pre-indexing phase. This one is further
   *            initialized in the indexing phase using passed closure.
   * @param isPreIndexingPhase
   *            whether the method is called in a pre-indexing phase, i.e.
   *            when the global index is not yet fully updated. You must not
   *            rely on linking using the index if isPreIndexingPhase is
   *            <code>true</code>.
   */
  protected void _infer(final MdModel model, final IJvmDeclaredTypeAcceptor acceptor, final boolean isPreIndexingPhase) {
    if (((model.getName() == null) || (model.getBundle() == null))) {
      return;
    }
    final MdBundle bundle = model.getBundle();
    String _qualifiedTargetClass = this.getQualifiedTargetClass(bundle);
    JvmGenericType _class = this._jvmTypesBuilder.toClass(bundle, _qualifiedTargetClass);
    final Procedure1<JvmGenericType> _function = new Procedure1<JvmGenericType>() {
      @Override
      public void apply(final JvmGenericType it) {
        EList<JvmTypeReference> _superTypes = it.getSuperTypes();
        JvmTypeReference _typeRef = MetaDataJvmModelInferrer.this._typeReferenceBuilder.typeRef("org.eclipse.elk.core.data.ILayoutMetaDataProvider");
        MetaDataJvmModelInferrer.this._jvmTypesBuilder.<JvmTypeReference>operator_add(_superTypes, _typeRef);
        String _documentation = MetaDataJvmModelInferrer.this._jvmTypesBuilder.getDocumentation(model);
        MetaDataJvmModelInferrer.this._jvmTypesBuilder.setFileHeader(it, _documentation);
        String _documentation_1 = MetaDataJvmModelInferrer.this._jvmTypesBuilder.getDocumentation(bundle);
        MetaDataJvmModelInferrer.this._jvmTypesBuilder.setDocumentation(it, _documentation_1);
        EList<MdBundleMember> _members = bundle.getMembers();
        Iterable<MdOption> _allOptionDefinitions = MetaDataJvmModelInferrer.this.getAllOptionDefinitions(_members);
        for (final MdOption property : _allOptionDefinitions) {
          {
            final JvmField constant = MetaDataJvmModelInferrer.this.toOptionConstant(property);
            XExpression _defaultValue = property.getDefaultValue();
            boolean _tripleNotEquals = (_defaultValue != null);
            if (_tripleNotEquals) {
              EList<JvmMember> _members_1 = it.getMembers();
              JvmField _optionDefault = MetaDataJvmModelInferrer.this.toOptionDefault(property);
              MetaDataJvmModelInferrer.this._jvmTypesBuilder.<JvmField>operator_add(_members_1, _optionDefault);
            }
            XExpression _lowerBound = property.getLowerBound();
            boolean _tripleNotEquals_1 = (_lowerBound != null);
            if (_tripleNotEquals_1) {
              EList<JvmMember> _members_2 = it.getMembers();
              JvmField _optionLowerBound = MetaDataJvmModelInferrer.this.toOptionLowerBound(property);
              MetaDataJvmModelInferrer.this._jvmTypesBuilder.<JvmField>operator_add(_members_2, _optionLowerBound);
            }
            XExpression _upperBound = property.getUpperBound();
            boolean _tripleNotEquals_2 = (_upperBound != null);
            if (_tripleNotEquals_2) {
              EList<JvmMember> _members_3 = it.getMembers();
              JvmField _optionUpperBound = MetaDataJvmModelInferrer.this.toOptionUpperBound(property);
              MetaDataJvmModelInferrer.this._jvmTypesBuilder.<JvmField>operator_add(_members_3, _optionUpperBound);
            }
            EList<JvmMember> _members_4 = it.getMembers();
            MetaDataJvmModelInferrer.this._jvmTypesBuilder.<JvmField>operator_add(_members_4, constant);
          }
        }
        EList<MdBundleMember> _members_1 = bundle.getMembers();
        Iterable<MdOption> _allOptionDefinitions_1 = MetaDataJvmModelInferrer.this.getAllOptionDefinitions(_members_1);
        for (final MdOption property_1 : _allOptionDefinitions_1) {
          EList<MdOptionDependency> _dependencies = property_1.getDependencies();
          for (final MdOptionDependency dependency : _dependencies) {
            XExpression _value = dependency.getValue();
            boolean _tripleNotEquals = (_value != null);
            if (_tripleNotEquals) {
              EList<JvmMember> _members_2 = it.getMembers();
              JvmField _dependencyValue = MetaDataJvmModelInferrer.this.toDependencyValue(dependency);
              MetaDataJvmModelInferrer.this._jvmTypesBuilder.<JvmField>operator_add(_members_2, _dependencyValue);
            }
          }
        }
        EList<JvmMember> _members_3 = it.getMembers();
        JvmTypeReference _typeRef_1 = MetaDataJvmModelInferrer.this._typeReferenceBuilder.typeRef(void.class);
        final Procedure1<JvmOperation> _function = new Procedure1<JvmOperation>() {
          @Override
          public void apply(final JvmOperation it) {
            EList<JvmFormalParameter> _parameters = it.getParameters();
            JvmTypeReference _typeRef = MetaDataJvmModelInferrer.this._typeReferenceBuilder.typeRef("org.eclipse.elk.core.data.ILayoutMetaDataProvider.Registry");
            JvmFormalParameter _parameter = MetaDataJvmModelInferrer.this._jvmTypesBuilder.toParameter(bundle, "registry", _typeRef);
            MetaDataJvmModelInferrer.this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _parameter);
            StringConcatenationClient _client = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                StringConcatenationClient _registerLayoutOptions = MetaDataJvmModelInferrer.this.registerLayoutOptions(bundle);
                _builder.append(_registerLayoutOptions, "");
                _builder.newLineIfNotEmpty();
                StringConcatenationClient _registerLayoutCategories = MetaDataJvmModelInferrer.this.registerLayoutCategories(bundle);
                _builder.append(_registerLayoutCategories, "");
                _builder.newLineIfNotEmpty();
                StringConcatenationClient _registerLayoutAlgorithms = MetaDataJvmModelInferrer.this.registerLayoutAlgorithms(bundle);
                _builder.append(_registerLayoutAlgorithms, "");
                _builder.newLineIfNotEmpty();
              }
            };
            MetaDataJvmModelInferrer.this._jvmTypesBuilder.setBody(it, _client);
          }
        };
        JvmOperation _method = MetaDataJvmModelInferrer.this._jvmTypesBuilder.toMethod(bundle, "apply", _typeRef_1, _function);
        MetaDataJvmModelInferrer.this._jvmTypesBuilder.<JvmOperation>operator_add(_members_3, _method);
      }
    };
    acceptor.<JvmGenericType>accept(_class, _function);
    EList<MdBundleMember> _members = bundle.getMembers();
    Iterable<MdAlgorithm> _filter = Iterables.<MdAlgorithm>filter(_members, MdAlgorithm.class);
    for (final MdAlgorithm algorithm : _filter) {
      this.inferAlgorithm(algorithm, acceptor, isPreIndexingPhase);
    }
  }
  
  private JvmField toOptionConstant(final MdOption option) {
    String _constantName = this.getConstantName(option);
    JvmTypeReference _elvis = null;
    JvmTypeReference _type = option.getType();
    if (_type != null) {
      _elvis = _type;
    } else {
      JvmTypeReference _typeRef = this._typeReferenceBuilder.typeRef(Object.class);
      _elvis = _typeRef;
    }
    JvmTypeReference _typeRef_1 = this._typeReferenceBuilder.typeRef(IProperty.class, _elvis);
    final Procedure1<JvmField> _function = new Procedure1<JvmField>() {
      @Override
      public void apply(final JvmField it) {
        it.setVisibility(JvmVisibility.PUBLIC);
        it.setStatic(true);
        it.setFinal(true);
        boolean _isDeprecated = option.isDeprecated();
        it.setDeprecated(_isDeprecated);
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("new ");
            _builder.append(Property.class, "");
            _builder.append("<");
            JvmTypeReference _elvis = null;
            JvmTypeReference _type = option.getType();
            JvmTypeReference _asWrapperTypeIfPrimitive = MetaDataJvmModelInferrer.this._primitives.asWrapperTypeIfPrimitive(_type);
            if (_asWrapperTypeIfPrimitive != null) {
              _elvis = _asWrapperTypeIfPrimitive;
            } else {
              JvmTypeReference _typeRef = MetaDataJvmModelInferrer.this._typeReferenceBuilder.typeRef(Object.class);
              _elvis = _typeRef;
            }
            _builder.append(_elvis, "");
            _builder.append(">(");
            _builder.newLineIfNotEmpty();
            _builder.append("        ");
            String _qualifiedName = MetaDataJvmModelInferrer.this.getQualifiedName(option);
            String _codeString = MetaDataJvmModelInferrer.this.toCodeString(_qualifiedName);
            _builder.append(_codeString, "        ");
            {
              boolean _hasDefaultOrBounds = MetaDataJvmModelInferrer.this.hasDefaultOrBounds(option);
              if (_hasDefaultOrBounds) {
                _builder.append(",");
                _builder.newLineIfNotEmpty();
                _builder.append("        ");
                {
                  XExpression _defaultValue = option.getDefaultValue();
                  boolean _tripleNotEquals = (_defaultValue != null);
                  if (_tripleNotEquals) {
                    String _defaultConstantName = MetaDataJvmModelInferrer.this.getDefaultConstantName(option);
                    _builder.append(_defaultConstantName, "        ");
                  } else {
                    _builder.append("null");
                  }
                }
                _builder.append(",");
                _builder.newLineIfNotEmpty();
                _builder.append("        ");
                {
                  XExpression _lowerBound = option.getLowerBound();
                  boolean _tripleNotEquals_1 = (_lowerBound != null);
                  if (_tripleNotEquals_1) {
                    String _lowerBoundConstantName = MetaDataJvmModelInferrer.this.getLowerBoundConstantName(option);
                    _builder.append(_lowerBoundConstantName, "        ");
                  } else {
                    _builder.append("null");
                  }
                }
                _builder.append(",");
                _builder.newLineIfNotEmpty();
                _builder.append("        ");
                {
                  XExpression _upperBound = option.getUpperBound();
                  boolean _tripleNotEquals_2 = (_upperBound != null);
                  if (_tripleNotEquals_2) {
                    String _upperBoundConstantName = MetaDataJvmModelInferrer.this.getUpperBoundConstantName(option);
                    _builder.append(_upperBoundConstantName, "        ");
                  } else {
                    _builder.append("null");
                  }
                }
              }
            }
            _builder.append(")");
          }
        };
        MetaDataJvmModelInferrer.this._jvmTypesBuilder.setInitializer(it, _client);
        String _description = option.getDescription();
        String _trimLines = MetaDataJvmModelInferrer.this.trimLines(_description);
        MetaDataJvmModelInferrer.this._jvmTypesBuilder.setDocumentation(it, _trimLines);
      }
    };
    return this._jvmTypesBuilder.toField(option, _constantName, _typeRef_1, _function);
  }
  
  private boolean hasDefaultOrBounds(final MdOption option) {
    return (((option.getDefaultValue() != null) || (option.getLowerBound() != null)) || (option.getUpperBound() != null));
  }
  
  private JvmField toOptionDefault(final MdOption option) {
    JvmTypeReference _elvis = null;
    JvmTypeReference _type = option.getType();
    JvmTypeReference _cloneWithProxies = this._jvmTypesBuilder.cloneWithProxies(_type);
    if (_cloneWithProxies != null) {
      _elvis = _cloneWithProxies;
    } else {
      JvmTypeReference _typeRef = this._typeReferenceBuilder.typeRef(Object.class);
      _elvis = _typeRef;
    }
    final JvmTypeReference optionType = _elvis;
    String _defaultConstantName = this.getDefaultConstantName(option);
    final Procedure1<JvmField> _function = new Procedure1<JvmField>() {
      @Override
      public void apply(final JvmField it) {
        it.setVisibility(JvmVisibility.PRIVATE);
        it.setStatic(true);
        it.setFinal(true);
        XExpression _defaultValue = option.getDefaultValue();
        MetaDataJvmModelInferrer.this._jvmTypesBuilder.setInitializer(it, _defaultValue);
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Default value for {@link #");
        String _constantName = MetaDataJvmModelInferrer.this.getConstantName(option);
        _builder.append(_constantName, "");
        _builder.append("}.");
        MetaDataJvmModelInferrer.this._jvmTypesBuilder.setDocumentation(it, _builder.toString());
      }
    };
    return this._jvmTypesBuilder.toField(option, _defaultConstantName, optionType, _function);
  }
  
  private JvmField toOptionLowerBound(final MdOption option) {
    JvmTypeReference _elvis = null;
    JvmTypeReference _type = option.getType();
    JvmTypeReference _asWrapperTypeIfPrimitive = this._primitives.asWrapperTypeIfPrimitive(_type);
    JvmTypeReference _wildcardSuper = null;
    if (_asWrapperTypeIfPrimitive!=null) {
      _wildcardSuper=this._typeReferenceBuilder.wildcardSuper(_asWrapperTypeIfPrimitive);
    }
    if (_wildcardSuper != null) {
      _elvis = _wildcardSuper;
    } else {
      JvmTypeReference _wildcard = this._typeReferenceBuilder.wildcard();
      _elvis = _wildcard;
    }
    final JvmTypeReference optionType = this._typeReferenceBuilder.typeRef(Comparable.class, _elvis);
    String _lowerBoundConstantName = this.getLowerBoundConstantName(option);
    final Procedure1<JvmField> _function = new Procedure1<JvmField>() {
      @Override
      public void apply(final JvmField it) {
        it.setVisibility(JvmVisibility.PRIVATE);
        it.setStatic(true);
        it.setFinal(true);
        XExpression _lowerBound = option.getLowerBound();
        MetaDataJvmModelInferrer.this._jvmTypesBuilder.setInitializer(it, _lowerBound);
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Lower bound value for {@link #");
        String _constantName = MetaDataJvmModelInferrer.this.getConstantName(option);
        _builder.append(_constantName, "");
        _builder.append("}.");
        MetaDataJvmModelInferrer.this._jvmTypesBuilder.setDocumentation(it, _builder.toString());
      }
    };
    return this._jvmTypesBuilder.toField(option, _lowerBoundConstantName, optionType, _function);
  }
  
  private JvmField toOptionUpperBound(final MdOption option) {
    JvmTypeReference _elvis = null;
    JvmTypeReference _type = option.getType();
    JvmTypeReference _asWrapperTypeIfPrimitive = this._primitives.asWrapperTypeIfPrimitive(_type);
    JvmTypeReference _wildcardSuper = null;
    if (_asWrapperTypeIfPrimitive!=null) {
      _wildcardSuper=this._typeReferenceBuilder.wildcardSuper(_asWrapperTypeIfPrimitive);
    }
    if (_wildcardSuper != null) {
      _elvis = _wildcardSuper;
    } else {
      JvmTypeReference _wildcard = this._typeReferenceBuilder.wildcard();
      _elvis = _wildcard;
    }
    final JvmTypeReference optionType = this._typeReferenceBuilder.typeRef(Comparable.class, _elvis);
    String _upperBoundConstantName = this.getUpperBoundConstantName(option);
    final Procedure1<JvmField> _function = new Procedure1<JvmField>() {
      @Override
      public void apply(final JvmField it) {
        it.setVisibility(JvmVisibility.PRIVATE);
        it.setStatic(true);
        it.setFinal(true);
        XExpression _upperBound = option.getUpperBound();
        MetaDataJvmModelInferrer.this._jvmTypesBuilder.setInitializer(it, _upperBound);
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Upper bound value for {@link #");
        String _constantName = MetaDataJvmModelInferrer.this.getConstantName(option);
        _builder.append(_constantName, "");
        _builder.append("}.");
        MetaDataJvmModelInferrer.this._jvmTypesBuilder.setDocumentation(it, _builder.toString());
      }
    };
    return this._jvmTypesBuilder.toField(option, _upperBoundConstantName, optionType, _function);
  }
  
  private JvmField toDependencyValue(final MdOptionDependency dependency) {
    EObject _eContainer = dependency.eContainer();
    final MdOption source = ((MdOption) _eContainer);
    JvmTypeReference _elvis = null;
    MdOption _target = dependency.getTarget();
    JvmTypeReference _type = _target.getType();
    JvmTypeReference _cloneWithProxies = this._jvmTypesBuilder.cloneWithProxies(_type);
    if (_cloneWithProxies != null) {
      _elvis = _cloneWithProxies;
    } else {
      JvmTypeReference _typeRef = this._typeReferenceBuilder.typeRef(Object.class);
      _elvis = _typeRef;
    }
    final JvmTypeReference optionType = _elvis;
    String _dependencyConstantName = this.getDependencyConstantName(dependency);
    final Procedure1<JvmField> _function = new Procedure1<JvmField>() {
      @Override
      public void apply(final JvmField it) {
        it.setVisibility(JvmVisibility.PRIVATE);
        it.setStatic(true);
        it.setFinal(true);
        XExpression _value = dependency.getValue();
        MetaDataJvmModelInferrer.this._jvmTypesBuilder.setInitializer(it, _value);
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Required value for dependency between {@link #");
        String _constantName = MetaDataJvmModelInferrer.this.getConstantName(source);
        _builder.append(_constantName, "");
        _builder.append("} and {@link #");
        MdOption _target = dependency.getTarget();
        String _constantName_1 = MetaDataJvmModelInferrer.this.getConstantName(_target);
        _builder.append(_constantName_1, "");
        _builder.append("}.");
        MetaDataJvmModelInferrer.this._jvmTypesBuilder.setDocumentation(it, _builder.toString());
      }
    };
    return this._jvmTypesBuilder.toField(dependency, _dependencyConstantName, optionType, _function);
  }
  
  private StringConcatenationClient registerLayoutOptions(final MdBundle bundle) {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        {
          EList<MdBundleMember> _members = bundle.getMembers();
          Iterable<MdOption> _allOptionDefinitions = MetaDataJvmModelInferrer.this.getAllOptionDefinitions(_members);
          for(final MdOption option : _allOptionDefinitions) {
            _builder.append("registry.register(new org.eclipse.elk.core.data.LayoutOptionData(");
            _builder.newLine();
            _builder.append("    ");
            String _qualifiedName = MetaDataJvmModelInferrer.this.getQualifiedName(option);
            String _codeString = MetaDataJvmModelInferrer.this.toCodeString(_qualifiedName);
            _builder.append(_codeString, "    ");
            _builder.append(",");
            _builder.newLineIfNotEmpty();
            _builder.append("    ");
            Iterable<MdGroup> _groups = MetaDataJvmModelInferrer.this.getGroups(option);
            final Function1<MdGroup, String> _function = new Function1<MdGroup, String>() {
              @Override
              public String apply(final MdGroup it) {
                return it.getName();
              }
            };
            Iterable<String> _map = IterableExtensions.<MdGroup, String>map(_groups, _function);
            String _join = IterableExtensions.join(_map, ".");
            String _codeString_1 = MetaDataJvmModelInferrer.this.toCodeString(_join);
            _builder.append(_codeString_1, "    ");
            _builder.append(",");
            _builder.newLineIfNotEmpty();
            _builder.append("    ");
            String _elvis = null;
            String _label = option.getLabel();
            if (_label != null) {
              _elvis = _label;
            } else {
              String _name = option.getName();
              _elvis = _name;
            }
            String _shrinkWhiteSpace = MetaDataJvmModelInferrer.this.shrinkWhiteSpace(_elvis);
            String _codeString_2 = MetaDataJvmModelInferrer.this.toCodeString(_shrinkWhiteSpace);
            _builder.append(_codeString_2, "    ");
            _builder.append(",");
            _builder.newLineIfNotEmpty();
            _builder.append("    ");
            String _description = option.getDescription();
            String _shrinkWhiteSpace_1 = MetaDataJvmModelInferrer.this.shrinkWhiteSpace(_description);
            String _codeString_3 = MetaDataJvmModelInferrer.this.toCodeString(_shrinkWhiteSpace_1);
            _builder.append(_codeString_3, "    ");
            _builder.append(",");
            _builder.newLineIfNotEmpty();
            _builder.append("    ");
            {
              XExpression _defaultValue = option.getDefaultValue();
              boolean _tripleEquals = (_defaultValue == null);
              if (_tripleEquals) {
                _builder.append("null,");
              } else {
                String _defaultConstantName = MetaDataJvmModelInferrer.this.getDefaultConstantName(option);
                _builder.append(_defaultConstantName, "    ");
                _builder.append(",");
              }
            }
            _builder.newLineIfNotEmpty();
            _builder.append("    ");
            {
              XExpression _lowerBound = option.getLowerBound();
              boolean _tripleEquals_1 = (_lowerBound == null);
              if (_tripleEquals_1) {
                _builder.append("null,");
              } else {
                String _lowerBoundConstantName = MetaDataJvmModelInferrer.this.getLowerBoundConstantName(option);
                _builder.append(_lowerBoundConstantName, "    ");
                _builder.append(",");
              }
            }
            _builder.newLineIfNotEmpty();
            _builder.append("    ");
            {
              XExpression _upperBound = option.getUpperBound();
              boolean _tripleEquals_2 = (_upperBound == null);
              if (_tripleEquals_2) {
                _builder.append("null,");
              } else {
                String _upperBoundConstantName = MetaDataJvmModelInferrer.this.getUpperBoundConstantName(option);
                _builder.append(_upperBoundConstantName, "    ");
                _builder.append(",");
              }
            }
            _builder.newLineIfNotEmpty();
            _builder.append("    ");
            _builder.append("org.eclipse.elk.core.data.LayoutOptionData.Type.");
            String _optionType = MetaDataJvmModelInferrer.this.getOptionType(option);
            _builder.append(_optionType, "    ");
            _builder.append(",");
            _builder.newLineIfNotEmpty();
            _builder.append("    ");
            JvmTypeReference _optionTypeClass = MetaDataJvmModelInferrer.this.getOptionTypeClass(option);
            _builder.append(_optionTypeClass, "    ");
            _builder.append(".class,");
            _builder.newLineIfNotEmpty();
            {
              EList<MdOptionTargetType> _targets = option.getTargets();
              boolean _isEmpty = _targets.isEmpty();
              if (_isEmpty) {
                _builder.append("    ");
                _builder.append("null,");
                _builder.newLine();
              } else {
                _builder.append("    ");
                _builder.append(EnumSet.class, "    ");
                _builder.append(".of(");
                {
                  EList<MdOptionTargetType> _targets_1 = option.getTargets();
                  boolean _hasElements = false;
                  for(final MdOptionTargetType t : _targets_1) {
                    if (!_hasElements) {
                      _hasElements = true;
                    } else {
                      _builder.appendImmediate(", ", "    ");
                    }
                    _builder.append("org.eclipse.elk.core.data.LayoutOptionData.Target.");
                    String _string = t.toString();
                    String _upperCase = _string.toUpperCase();
                    _builder.append(_upperCase, "    ");
                  }
                }
                _builder.append("),");
                _builder.newLineIfNotEmpty();
              }
            }
            {
              if (((option.isProgrammatic() || option.isOutput()) || option.isGlobal())) {
                _builder.append("    ");
                _builder.append("org.eclipse.elk.core.data.LayoutOptionData.Visibility.HIDDEN");
                _builder.newLine();
              } else {
                boolean _isAdvanced = option.isAdvanced();
                if (_isAdvanced) {
                  _builder.append("    ");
                  _builder.append("org.eclipse.elk.core.data.LayoutOptionData.Visibility.ADVANCED");
                  _builder.newLine();
                } else {
                  _builder.append("    ");
                  _builder.append("org.eclipse.elk.core.data.LayoutOptionData.Visibility.VISIBLE");
                  _builder.newLine();
                }
              }
            }
            {
              EList<String> _legacyIds = option.getLegacyIds();
              boolean _isEmpty_1 = _legacyIds.isEmpty();
              boolean _not = (!_isEmpty_1);
              if (_not) {
                _builder.append("    ");
                EList<String> _legacyIds_1 = option.getLegacyIds();
                final Function1<String, String> _function_1 = new Function1<String, String>() {
                  @Override
                  public String apply(final String it) {
                    return ((", \"" + it) + "\"");
                  }
                };
                List<String> _map_1 = ListExtensions.<String, String>map(_legacyIds_1, _function_1);
                String _join_1 = IterableExtensions.join(_map_1);
                _builder.append(_join_1, "    ");
                _builder.newLineIfNotEmpty();
              }
            }
            _builder.append("));");
            _builder.newLine();
            {
              EList<MdOptionDependency> _dependencies = option.getDependencies();
              for(final MdOptionDependency dependency : _dependencies) {
                _builder.append("registry.addDependency(");
                _builder.newLine();
                _builder.append("    ");
                String _qualifiedName_1 = MetaDataJvmModelInferrer.this.getQualifiedName(option);
                String _codeString_4 = MetaDataJvmModelInferrer.this.toCodeString(_qualifiedName_1);
                _builder.append(_codeString_4, "    ");
                _builder.append(",");
                _builder.newLineIfNotEmpty();
                _builder.append("    ");
                MdOption _target = dependency.getTarget();
                String _qualifiedName_2 = MetaDataJvmModelInferrer.this.getQualifiedName(_target);
                String _codeString_5 = MetaDataJvmModelInferrer.this.toCodeString(_qualifiedName_2);
                _builder.append(_codeString_5, "    ");
                _builder.append(",");
                _builder.newLineIfNotEmpty();
                {
                  XExpression _value = dependency.getValue();
                  boolean _tripleEquals_3 = (_value == null);
                  if (_tripleEquals_3) {
                    _builder.append("    ");
                    _builder.append("null");
                    _builder.newLine();
                  } else {
                    _builder.append("    ");
                    String _dependencyConstantName = MetaDataJvmModelInferrer.this.getDependencyConstantName(dependency);
                    _builder.append(_dependencyConstantName, "    ");
                    _builder.newLineIfNotEmpty();
                  }
                }
                _builder.append(");");
                _builder.newLine();
              }
            }
          }
        }
      }
    };
    return _client;
  }
  
  private StringConcatenationClient registerLayoutCategories(final MdBundle bundle) {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        {
          EList<MdBundleMember> _members = bundle.getMembers();
          Iterable<MdCategory> _filter = Iterables.<MdCategory>filter(_members, MdCategory.class);
          for(final MdCategory category : _filter) {
            _builder.append("registry.register(new org.eclipse.elk.core.data.LayoutCategoryData(");
            _builder.newLine();
            _builder.append("    ");
            String _qualifiedName = MetaDataJvmModelInferrer.this.getQualifiedName(category);
            String _codeString = MetaDataJvmModelInferrer.this.toCodeString(_qualifiedName);
            _builder.append(_codeString, "    ");
            _builder.append(",");
            _builder.newLineIfNotEmpty();
            _builder.append("    ");
            String _elvis = null;
            String _label = category.getLabel();
            if (_label != null) {
              _elvis = _label;
            } else {
              String _name = category.getName();
              _elvis = _name;
            }
            String _shrinkWhiteSpace = MetaDataJvmModelInferrer.this.shrinkWhiteSpace(_elvis);
            String _codeString_1 = MetaDataJvmModelInferrer.this.toCodeString(_shrinkWhiteSpace);
            _builder.append(_codeString_1, "    ");
            _builder.append(",");
            _builder.newLineIfNotEmpty();
            _builder.append("    ");
            String _description = category.getDescription();
            String _shrinkWhiteSpace_1 = MetaDataJvmModelInferrer.this.shrinkWhiteSpace(_description);
            String _codeString_2 = MetaDataJvmModelInferrer.this.toCodeString(_shrinkWhiteSpace_1);
            _builder.append(_codeString_2, "    ");
            _builder.newLineIfNotEmpty();
            _builder.append("));");
            _builder.newLine();
          }
        }
      }
    };
    return _client;
  }
  
  private StringConcatenationClient registerLayoutAlgorithms(final MdBundle bundle) {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        {
          EList<MdBundleMember> _members = bundle.getMembers();
          Iterable<MdAlgorithm> _filter = Iterables.<MdAlgorithm>filter(_members, MdAlgorithm.class);
          for(final MdAlgorithm algorithm : _filter) {
            _builder.append("new ");
            String _qualifiedTargetClass = MetaDataJvmModelInferrer.this.getQualifiedTargetClass(algorithm);
            _builder.append(_qualifiedTargetClass, "");
            _builder.append("().apply(registry);");
            _builder.newLineIfNotEmpty();
          }
        }
      }
    };
    return _client;
  }
  
  /**
   * Generates code for the given algorithm'm metadata provider class.
   * 
   * @param algorithm
   *            the algorithm to create one or more {@link JvmDeclaredType declared types} from.
   * @param acceptor
   *            each created {@link JvmDeclaredType type} without a container should be passed to the acceptor
   *            in order to get attached to the current resource. The acceptor's
   *            {@link IJvmDeclaredTypeAcceptor#accept(org.eclipse.xtext.common.types.JvmDeclaredType) accept(..)}
   *            method takes the constructed empty type for the pre-indexing phase. This one is further
   *            initialized in the indexing phase using passed closure.
   * @param isPreIndexingPhase
   *            whether the method is called in a pre-indexing phase, i.e.
   *            when the global index is not yet fully updated. You must not
   *            rely on linking using the index if isPreIndexingPhase is
   *            <code>true</code>.
   */
  private void inferAlgorithm(final MdAlgorithm algorithm, final IJvmDeclaredTypeAcceptor acceptor, final boolean isPreIndexingPhase) {
    String _name = algorithm.getName();
    boolean _tripleEquals = (_name == null);
    if (_tripleEquals) {
      return;
    }
    String _qualifiedTargetClass = this.getQualifiedTargetClass(algorithm);
    JvmGenericType _class = this._jvmTypesBuilder.toClass(algorithm, _qualifiedTargetClass);
    final Procedure1<JvmGenericType> _function = new Procedure1<JvmGenericType>() {
      @Override
      public void apply(final JvmGenericType it) {
        EList<JvmTypeReference> _superTypes = it.getSuperTypes();
        JvmTypeReference _typeRef = MetaDataJvmModelInferrer.this._typeReferenceBuilder.typeRef("org.eclipse.elk.core.data.ILayoutMetaDataProvider");
        MetaDataJvmModelInferrer.this._jvmTypesBuilder.<JvmTypeReference>operator_add(_superTypes, _typeRef);
        MdBundle _bundle = MetaDataJvmModelInferrer.this.getBundle(algorithm);
        String _documentation = MetaDataJvmModelInferrer.this._jvmTypesBuilder.getDocumentation(_bundle);
        MetaDataJvmModelInferrer.this._jvmTypesBuilder.setFileHeader(it, _documentation);
        String _documentation_1 = algorithm.getDocumentation();
        MetaDataJvmModelInferrer.this._jvmTypesBuilder.setDocumentation(it, _documentation_1);
        EList<MdOptionSupport> _supportedOptions = algorithm.getSupportedOptions();
        for (final MdOptionSupport support : _supportedOptions) {
          {
            XExpression _value = support.getValue();
            boolean _tripleNotEquals = (_value != null);
            if (_tripleNotEquals) {
              EList<JvmMember> _members = it.getMembers();
              JvmField _supportedOptionDefault = MetaDataJvmModelInferrer.this.toSupportedOptionDefault(support);
              MetaDataJvmModelInferrer.this._jvmTypesBuilder.<JvmField>operator_add(_members, _supportedOptionDefault);
            }
            EList<JvmMember> _members_1 = it.getMembers();
            JvmField _supportedOptionConstant = MetaDataJvmModelInferrer.this.toSupportedOptionConstant(support);
            MetaDataJvmModelInferrer.this._jvmTypesBuilder.<JvmField>operator_add(_members_1, _supportedOptionConstant);
          }
        }
        EList<JvmMember> _members = it.getMembers();
        JvmTypeReference _typeRef_1 = MetaDataJvmModelInferrer.this._typeReferenceBuilder.typeRef(void.class);
        final Procedure1<JvmOperation> _function = new Procedure1<JvmOperation>() {
          @Override
          public void apply(final JvmOperation it) {
            EList<JvmFormalParameter> _parameters = it.getParameters();
            JvmTypeReference _typeRef = MetaDataJvmModelInferrer.this._typeReferenceBuilder.typeRef("org.eclipse.elk.core.data.ILayoutMetaDataProvider.Registry");
            JvmFormalParameter _parameter = MetaDataJvmModelInferrer.this._jvmTypesBuilder.toParameter(algorithm, "registry", _typeRef);
            MetaDataJvmModelInferrer.this._jvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _parameter);
            StringConcatenationClient _client = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                StringConcatenationClient _registerLayoutAlgorithm = MetaDataJvmModelInferrer.this.registerLayoutAlgorithm(algorithm);
                _builder.append(_registerLayoutAlgorithm, "");
                _builder.newLineIfNotEmpty();
              }
            };
            MetaDataJvmModelInferrer.this._jvmTypesBuilder.setBody(it, _client);
          }
        };
        JvmOperation _method = MetaDataJvmModelInferrer.this._jvmTypesBuilder.toMethod(algorithm, "apply", _typeRef_1, _function);
        MetaDataJvmModelInferrer.this._jvmTypesBuilder.<JvmOperation>operator_add(_members, _method);
      }
    };
    acceptor.<JvmGenericType>accept(_class, _function);
  }
  
  private JvmField toSupportedOptionDefault(final MdOptionSupport support) {
    EObject _eContainer = support.eContainer();
    final MdAlgorithm algorithm = ((MdAlgorithm) _eContainer);
    JvmTypeReference _elvis = null;
    MdOption _option = support.getOption();
    JvmTypeReference _type = _option.getType();
    JvmTypeReference _cloneWithProxies = this._jvmTypesBuilder.cloneWithProxies(_type);
    if (_cloneWithProxies != null) {
      _elvis = _cloneWithProxies;
    } else {
      JvmTypeReference _typeRef = this._typeReferenceBuilder.typeRef(Object.class);
      _elvis = _typeRef;
    }
    final JvmTypeReference optionType = _elvis;
    MdOption _option_1 = support.getOption();
    String _defaultConstantName = this.getDefaultConstantName(_option_1);
    final Procedure1<JvmField> _function = new Procedure1<JvmField>() {
      @Override
      public void apply(final JvmField it) {
        it.setVisibility(JvmVisibility.PRIVATE);
        it.setStatic(true);
        it.setFinal(true);
        XExpression _value = support.getValue();
        MetaDataJvmModelInferrer.this._jvmTypesBuilder.setInitializer(it, _value);
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Default value for {@link #");
        MdOption _option = support.getOption();
        String _constantName = MetaDataJvmModelInferrer.this.getConstantName(_option);
        _builder.append(_constantName, "");
        _builder.append("} with algorithm \"");
        String _elvis = null;
        String _label = algorithm.getLabel();
        if (_label != null) {
          _elvis = _label;
        } else {
          String _name = algorithm.getName();
          _elvis = _name;
        }
        _builder.append(_elvis, "");
        _builder.append("\".");
        MetaDataJvmModelInferrer.this._jvmTypesBuilder.setDocumentation(it, _builder.toString());
      }
    };
    return this._jvmTypesBuilder.toField(support, _defaultConstantName, optionType, _function);
  }
  
  private JvmField toSupportedOptionConstant(final MdOptionSupport support) {
    MdOption _option = support.getOption();
    String _constantName = this.getConstantName(_option);
    JvmTypeReference _elvis = null;
    MdOption _option_1 = support.getOption();
    JvmTypeReference _type = _option_1.getType();
    if (_type != null) {
      _elvis = _type;
    } else {
      JvmTypeReference _typeRef = this._typeReferenceBuilder.typeRef(Object.class);
      _elvis = _typeRef;
    }
    JvmTypeReference _typeRef_1 = this._typeReferenceBuilder.typeRef(IProperty.class, _elvis);
    final Procedure1<JvmField> _function = new Procedure1<JvmField>() {
      @Override
      public void apply(final JvmField it) {
        it.setVisibility(JvmVisibility.PUBLIC);
        it.setStatic(true);
        it.setFinal(true);
        MdOption _option = support.getOption();
        boolean _isDeprecated = _option.isDeprecated();
        it.setDeprecated(_isDeprecated);
        StringConcatenationClient _xifexpression = null;
        XExpression _value = support.getValue();
        boolean _tripleEquals = (_value == null);
        if (_tripleEquals) {
          StringConcatenationClient _client = new StringConcatenationClient() {
            @Override
            protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
              MdOption _option = support.getOption();
              MdBundle _bundle = MetaDataJvmModelInferrer.this.getBundle(_option);
              String _qualifiedTargetClass = MetaDataJvmModelInferrer.this.getQualifiedTargetClass(_bundle);
              JvmTypeReference _typeRef = MetaDataJvmModelInferrer.this._typeReferenceBuilder.typeRef(_qualifiedTargetClass);
              _builder.append(_typeRef, "");
              _builder.append(".");
              MdOption _option_1 = support.getOption();
              String _constantName = MetaDataJvmModelInferrer.this.getConstantName(_option_1);
              _builder.append(_constantName, "");
            }
          };
          _xifexpression = _client;
        } else {
          StringConcatenationClient _client_1 = new StringConcatenationClient() {
            @Override
            protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
              _builder.append("new ");
              _builder.append(Property.class, "");
              _builder.append("<");
              JvmTypeReference _elvis = null;
              MdOption _option = support.getOption();
              JvmTypeReference _type = _option.getType();
              JvmTypeReference _asWrapperTypeIfPrimitive = MetaDataJvmModelInferrer.this._primitives.asWrapperTypeIfPrimitive(_type);
              if (_asWrapperTypeIfPrimitive != null) {
                _elvis = _asWrapperTypeIfPrimitive;
              } else {
                JvmTypeReference _typeRef = MetaDataJvmModelInferrer.this._typeReferenceBuilder.typeRef(Object.class);
                _elvis = _typeRef;
              }
              _builder.append(_elvis, "");
              _builder.append(">(");
              _builder.newLineIfNotEmpty();
              _builder.append("                            ");
              MdOption _option_1 = support.getOption();
              MdBundle _bundle = MetaDataJvmModelInferrer.this.getBundle(_option_1);
              String _qualifiedTargetClass = MetaDataJvmModelInferrer.this.getQualifiedTargetClass(_bundle);
              JvmTypeReference _typeRef_1 = MetaDataJvmModelInferrer.this._typeReferenceBuilder.typeRef(_qualifiedTargetClass);
              _builder.append(_typeRef_1, "                            ");
              _builder.append(".");
              MdOption _option_2 = support.getOption();
              String _constantName = MetaDataJvmModelInferrer.this.getConstantName(_option_2);
              _builder.append(_constantName, "                            ");
              _builder.append(",");
              _builder.newLineIfNotEmpty();
              _builder.append("                            ");
              MdOption _option_3 = support.getOption();
              String _defaultConstantName = MetaDataJvmModelInferrer.this.getDefaultConstantName(_option_3);
              _builder.append(_defaultConstantName, "                            ");
              _builder.append(")");
            }
          };
          _xifexpression = _client_1;
        }
        MetaDataJvmModelInferrer.this._jvmTypesBuilder.setInitializer(it, _xifexpression);
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Property constant to access ");
        String _elvis = null;
        MdOption _option_1 = support.getOption();
        String _label = _option_1.getLabel();
        if (_label != null) {
          _elvis = _label;
        } else {
          MdOption _option_2 = support.getOption();
          String _name = _option_2.getName();
          _elvis = _name;
        }
        _builder.append(_elvis, "");
        _builder.append(" from within the layout algorithm code.");
        MetaDataJvmModelInferrer.this._jvmTypesBuilder.setDocumentation(it, _builder.toString());
      }
    };
    return this._jvmTypesBuilder.toField(support, _constantName, _typeRef_1, _function);
  }
  
  private StringConcatenationClient registerLayoutAlgorithm(final MdAlgorithm algorithm) {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        _builder.append("registry.register(new org.eclipse.elk.core.data.LayoutAlgorithmData(");
        _builder.newLine();
        _builder.append("    ");
        String _qualifiedName = MetaDataJvmModelInferrer.this.getQualifiedName(algorithm);
        String _codeString = MetaDataJvmModelInferrer.this.toCodeString(_qualifiedName);
        _builder.append(_codeString, "    ");
        _builder.append(",");
        _builder.newLineIfNotEmpty();
        _builder.append("    ");
        String _elvis = null;
        String _label = algorithm.getLabel();
        if (_label != null) {
          _elvis = _label;
        } else {
          String _name = algorithm.getName();
          _elvis = _name;
        }
        String _shrinkWhiteSpace = MetaDataJvmModelInferrer.this.shrinkWhiteSpace(_elvis);
        String _codeString_1 = MetaDataJvmModelInferrer.this.toCodeString(_shrinkWhiteSpace);
        _builder.append(_codeString_1, "    ");
        _builder.append(",");
        _builder.newLineIfNotEmpty();
        _builder.append("    ");
        String _description = algorithm.getDescription();
        String _shrinkWhiteSpace_1 = MetaDataJvmModelInferrer.this.shrinkWhiteSpace(_description);
        String _codeString_2 = MetaDataJvmModelInferrer.this.toCodeString(_shrinkWhiteSpace_1);
        _builder.append(_codeString_2, "    ");
        _builder.append(",");
        _builder.newLineIfNotEmpty();
        _builder.append("    ");
        _builder.append("new org.eclipse.elk.core.util.AlgorithmFactory(");
        JvmTypeReference _provider = algorithm.getProvider();
        _builder.append(_provider, "    ");
        _builder.append(".class, \"");
        String _parameter = algorithm.getParameter();
        _builder.append(_parameter, "    ");
        _builder.append("\"),");
        _builder.newLineIfNotEmpty();
        _builder.append("    ");
        MdCategory _category = algorithm.getCategory();
        String _qualifiedName_1 = null;
        if (_category!=null) {
          _qualifiedName_1=MetaDataJvmModelInferrer.this.getQualifiedName(_category);
        }
        String _codeString_3 = MetaDataJvmModelInferrer.this.toCodeString(_qualifiedName_1);
        _builder.append(_codeString_3, "    ");
        _builder.append(",");
        _builder.newLineIfNotEmpty();
        _builder.append("    ");
        MdBundle _bundle = MetaDataJvmModelInferrer.this.getBundle(algorithm);
        String _label_1 = null;
        if (_bundle!=null) {
          _label_1=_bundle.getLabel();
        }
        String _codeString_4 = MetaDataJvmModelInferrer.this.toCodeString(_label_1);
        _builder.append(_codeString_4, "    ");
        _builder.append(",");
        _builder.newLineIfNotEmpty();
        _builder.append("    ");
        String _previewImage = algorithm.getPreviewImage();
        String _codeString_5 = MetaDataJvmModelInferrer.this.toCodeString(_previewImage);
        _builder.append(_codeString_5, "    ");
        _builder.append(",");
        _builder.newLineIfNotEmpty();
        {
          EList<MdGraphFeature> _supportedFeatures = algorithm.getSupportedFeatures();
          boolean _isEmpty = _supportedFeatures.isEmpty();
          if (_isEmpty) {
            _builder.append("    ");
            _builder.append("null");
            _builder.newLine();
          } else {
            _builder.append("    ");
            _builder.append(EnumSet.class, "    ");
            _builder.append(".of(");
            {
              EList<MdGraphFeature> _supportedFeatures_1 = algorithm.getSupportedFeatures();
              boolean _hasElements = false;
              for(final MdGraphFeature f : _supportedFeatures_1) {
                if (!_hasElements) {
                  _hasElements = true;
                } else {
                  _builder.appendImmediate(", ", "    ");
                }
                _builder.append("org.eclipse.elk.core.options.GraphFeature.");
                String _string = f.toString();
                String _upperCase = _string.toUpperCase();
                _builder.append(_upperCase, "    ");
              }
            }
            _builder.append(")");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.append("));");
        _builder.newLine();
        {
          EList<MdOptionSupport> _supportedOptions = algorithm.getSupportedOptions();
          for(final MdOptionSupport support : _supportedOptions) {
            _builder.append("registry.addOptionSupport(");
            _builder.newLine();
            _builder.append("    ");
            String _qualifiedName_2 = MetaDataJvmModelInferrer.this.getQualifiedName(algorithm);
            String _codeString_6 = MetaDataJvmModelInferrer.this.toCodeString(_qualifiedName_2);
            _builder.append(_codeString_6, "    ");
            _builder.append(",");
            _builder.newLineIfNotEmpty();
            _builder.append("    ");
            MdOption _option = support.getOption();
            String _qualifiedName_3 = MetaDataJvmModelInferrer.this.getQualifiedName(_option);
            String _codeString_7 = MetaDataJvmModelInferrer.this.toCodeString(_qualifiedName_3);
            _builder.append(_codeString_7, "    ");
            _builder.append(",");
            _builder.newLineIfNotEmpty();
            {
              XExpression _value = support.getValue();
              boolean _tripleEquals = (_value == null);
              if (_tripleEquals) {
                _builder.append("    ");
                MdOption _option_1 = support.getOption();
                String _constantName = MetaDataJvmModelInferrer.this.getConstantName(_option_1);
                _builder.append(_constantName, "    ");
                _builder.append(".getDefault()");
                _builder.newLineIfNotEmpty();
              } else {
                _builder.append("    ");
                MdOption _option_2 = support.getOption();
                String _defaultConstantName = MetaDataJvmModelInferrer.this.getDefaultConstantName(_option_2);
                _builder.append(_defaultConstantName, "    ");
                _builder.newLineIfNotEmpty();
              }
            }
            _builder.append(");");
            _builder.newLine();
          }
        }
      }
    };
    return _client;
  }
  
  private Iterable<MdOption> getAllOptionDefinitions(final Iterable<? extends MdBundleMember> elements) {
    Iterable<MdOption> _filter = Iterables.<MdOption>filter(elements, MdOption.class);
    Iterable<MdGroup> _filter_1 = Iterables.<MdGroup>filter(elements, MdGroup.class);
    final Function1<MdGroup, Iterable<MdOption>> _function = new Function1<MdGroup, Iterable<MdOption>>() {
      @Override
      public Iterable<MdOption> apply(final MdGroup it) {
        EList<MdGroupOrOption> _children = it.getChildren();
        return MetaDataJvmModelInferrer.this.getAllOptionDefinitions(_children);
      }
    };
    Iterable<Iterable<MdOption>> _map = IterableExtensions.<MdGroup, Iterable<MdOption>>map(_filter_1, _function);
    Iterable<MdOption> _flatten = Iterables.<MdOption>concat(_map);
    return Iterables.<MdOption>concat(_filter, _flatten);
  }
  
  private String getQualifiedTargetClass(final MdBundle bundle) {
    EObject _eContainer = bundle.eContainer();
    final MdModel model = ((MdModel) _eContainer);
    String _elvis = null;
    String _targetClass = bundle.getTargetClass();
    if (_targetClass != null) {
      _elvis = _targetClass;
    } else {
      _elvis = "Metadata";
    }
    final String bundleClass = _elvis;
    String _name = model.getName();
    String _plus = (_name + ".");
    return (_plus + bundleClass);
  }
  
  private String getQualifiedTargetClass(final MdAlgorithm algorithm) {
    MdBundle _bundle = this.getBundle(algorithm);
    EObject _eContainer = _bundle.eContainer();
    final MdModel model = ((MdModel) _eContainer);
    String _elvis = null;
    String _targetClass = algorithm.getTargetClass();
    if (_targetClass != null) {
      _elvis = _targetClass;
    } else {
      String _name = algorithm.getName();
      String _firstUpper = StringExtensions.toFirstUpper(_name);
      String _plus = (_firstUpper + "Metadata");
      _elvis = _plus;
    }
    final String algorithmClass = _elvis;
    String _name_1 = model.getName();
    String _plus_1 = (_name_1 + ".");
    return (_plus_1 + algorithmClass);
  }
  
  private Iterable<MdGroup> getGroups(final MdBundleMember member) {
    LinkedList<MdGroup> _xblockexpression = null;
    {
      final LinkedList<MdGroup> groups = new LinkedList<MdGroup>();
      EObject group = member.eContainer();
      while ((group instanceof MdGroup)) {
        {
          groups.addFirst(((MdGroup)group));
          EObject _eContainer = ((MdGroup)group).eContainer();
          group = _eContainer;
        }
      }
      _xblockexpression = groups;
    }
    return _xblockexpression;
  }
  
  private String getOptionType(final MdOption option) {
    JvmTypeReference _type = option.getType();
    JvmType _type_1 = null;
    if (_type!=null) {
      _type_1=_type.getType();
    }
    final JvmType jvmType = _type_1;
    boolean _matched = false;
    if (jvmType instanceof JvmPrimitiveType) {
      _matched=true;
      String _identifier = ((JvmPrimitiveType)jvmType).getIdentifier();
      boolean _matched_1 = false;
      String _name = boolean.class.getName();
      if (Objects.equal(_identifier, _name)) {
        _matched_1=true;
        return "BOOLEAN";
      }
      if (!_matched_1) {
        String _name_1 = int.class.getName();
        if (Objects.equal(_identifier, _name_1)) {
          _matched_1=true;
          return "INT";
        }
      }
      if (!_matched_1) {
        String _name_2 = float.class.getName();
        if (Objects.equal(_identifier, _name_2)) {
          _matched_1=true;
          return "FLOAT";
        }
      }
      if (!_matched_1) {
        String _name_3 = double.class.getName();
        if (Objects.equal(_identifier, _name_3)) {
          _matched_1=true;
          return "FLOAT";
        }
      }
    }
    if (!_matched) {
      if (jvmType instanceof JvmGenericType) {
        _matched=true;
        String _identifier = ((JvmGenericType)jvmType).getIdentifier();
        boolean _matched_1 = false;
        String _canonicalName = Boolean.class.getCanonicalName();
        if (Objects.equal(_identifier, _canonicalName)) {
          _matched_1=true;
          return "BOOLEAN";
        }
        if (!_matched_1) {
          String _canonicalName_1 = Integer.class.getCanonicalName();
          if (Objects.equal(_identifier, _canonicalName_1)) {
            _matched_1=true;
            return "INT";
          }
        }
        if (!_matched_1) {
          String _canonicalName_2 = Float.class.getCanonicalName();
          if (Objects.equal(_identifier, _canonicalName_2)) {
            _matched_1=true;
            return "FLOAT";
          }
        }
        if (!_matched_1) {
          String _canonicalName_3 = Double.class.getCanonicalName();
          if (Objects.equal(_identifier, _canonicalName_3)) {
            _matched_1=true;
            return "FLOAT";
          }
        }
        if (!_matched_1) {
          String _canonicalName_4 = String.class.getCanonicalName();
          if (Objects.equal(_identifier, _canonicalName_4)) {
            _matched_1=true;
            return "STRING";
          }
        }
        if (!_matched_1) {
          String _canonicalName_5 = EnumSet.class.getCanonicalName();
          if (Objects.equal(_identifier, _canonicalName_5)) {
            _matched_1=true;
            return "ENUMSET";
          }
        }
        return "OBJECT";
      }
    }
    if (!_matched) {
      if (jvmType instanceof JvmEnumerationType) {
        _matched=true;
        return "ENUM";
      }
    }
    return "UNDEFINED";
  }
  
  private boolean hasSupertype(final JvmDeclaredType type, final Class<?> superType) {
    EList<JvmTypeReference> _superTypes = type.getSuperTypes();
    final Function1<JvmTypeReference, Boolean> _function = new Function1<JvmTypeReference, Boolean>() {
      @Override
      public Boolean apply(final JvmTypeReference t) {
        String _qualifiedName = t.getQualifiedName();
        String _canonicalName = superType.getCanonicalName();
        return Boolean.valueOf(Objects.equal(_qualifiedName, _canonicalName));
      }
    };
    JvmTypeReference _findFirst = IterableExtensions.<JvmTypeReference>findFirst(_superTypes, _function);
    boolean _notEquals = (!Objects.equal(_findFirst, null));
    if (_notEquals) {
      return true;
    } else {
      EList<JvmTypeReference> _superTypes_1 = type.getSuperTypes();
      final Function1<JvmTypeReference, JvmType> _function_1 = new Function1<JvmTypeReference, JvmType>() {
        @Override
        public JvmType apply(final JvmTypeReference rt) {
          return rt.getType();
        }
      };
      List<JvmType> _map = ListExtensions.<JvmTypeReference, JvmType>map(_superTypes_1, _function_1);
      Iterable<JvmDeclaredType> _filter = Iterables.<JvmDeclaredType>filter(_map, JvmDeclaredType.class);
      final Function1<JvmDeclaredType, Boolean> _function_2 = new Function1<JvmDeclaredType, Boolean>() {
        @Override
        public Boolean apply(final JvmDeclaredType t) {
          return Boolean.valueOf(MetaDataJvmModelInferrer.this.hasSupertype(t, superType));
        }
      };
      JvmDeclaredType _findFirst_1 = IterableExtensions.<JvmDeclaredType>findFirst(_filter, _function_2);
      return (!Objects.equal(_findFirst_1, null));
    }
  }
  
  private JvmTypeReference getOptionTypeClass(final MdOption property) {
    JvmTypeReference _xifexpression = null;
    JvmTypeReference _type = property.getType();
    boolean _notEquals = (!Objects.equal(_type, null));
    if (_notEquals) {
      JvmTypeReference _xblockexpression = null;
      {
        JvmTypeReference _type_1 = property.getType();
        JvmType _type_2 = _type_1.getType();
        if ((_type_2 instanceof JvmPrimitiveType)) {
          JvmTypeReference _type_3 = property.getType();
          JvmType _type_4 = _type_3.getType();
          final JvmPrimitiveType primitiveType = ((JvmPrimitiveType) _type_4);
          String _simpleName = primitiveType.getSimpleName();
          boolean _equals = Objects.equal(_simpleName, "double");
          if (_equals) {
            return this._typeReferenceBuilder.typeRef(Float.class);
          } else {
            String _simpleName_1 = primitiveType.getSimpleName();
            boolean _equals_1 = Objects.equal(_simpleName_1, "long");
            if (_equals_1) {
              return this._typeReferenceBuilder.typeRef(Integer.class);
            }
          }
        } else {
          JvmTypeReference _type_5 = property.getType();
          JvmType _type_6 = _type_5.getType();
          if ((_type_6 instanceof JvmGenericType)) {
            JvmTypeReference _type_7 = property.getType();
            JvmType _type_8 = _type_7.getType();
            final JvmGenericType genericType = ((JvmGenericType) _type_8);
            String _identifier = genericType.getIdentifier();
            boolean _equals_2 = Objects.equal(_identifier, "java.lang.Double");
            if (_equals_2) {
              return this._typeReferenceBuilder.typeRef(Float.class);
            } else {
              String _identifier_1 = genericType.getIdentifier();
              boolean _equals_3 = Objects.equal(_identifier_1, "java.lang.Long");
              if (_equals_3) {
                return this._typeReferenceBuilder.typeRef(Integer.class);
              } else {
                String _identifier_2 = genericType.getIdentifier();
                boolean _equals_4 = Objects.equal(_identifier_2, "java.util.EnumSet");
                if (_equals_4) {
                  JvmTypeReference _type_9 = property.getType();
                  final JvmParameterizedTypeReference outer = ((JvmParameterizedTypeReference) _type_9);
                  EList<JvmTypeReference> _arguments = outer.getArguments();
                  JvmTypeReference _head = IterableExtensions.<JvmTypeReference>head(_arguments);
                  return this._jvmTypesBuilder.cloneWithProxies(_head);
                }
              }
            }
          }
        }
        JvmTypeReference _type_10 = property.getType();
        JvmTypeReference _cloneWithProxies = this._jvmTypesBuilder.cloneWithProxies(_type_10);
        _xblockexpression = this._primitives.asWrapperTypeIfPrimitive(_cloneWithProxies);
      }
      _xifexpression = _xblockexpression;
    } else {
      return this._typeReferenceBuilder.typeRef(Void.class);
    }
    return _xifexpression;
  }
  
  private MdBundle getBundle(final MdBundleMember member) {
    EObject parent = member.eContainer();
    while ((!(parent instanceof MdBundle))) {
      EObject _eContainer = parent.eContainer();
      parent = _eContainer;
    }
    return ((MdBundle) parent);
  }
  
  private String getQualifiedName(final MdBundleMember member) {
    final MdBundle bundle = this.getBundle(member);
    EObject _eContainer = bundle.eContainer();
    final MdModel model = ((MdModel) _eContainer);
    String _elvis = null;
    String _idPrefix = bundle.getIdPrefix();
    if (_idPrefix != null) {
      _elvis = _idPrefix;
    } else {
      String _name = model.getName();
      _elvis = _name;
    }
    String prefix = _elvis;
    String _name_1 = member.getName();
    int _lastIndexOf = prefix.lastIndexOf(".");
    int _plus = (_lastIndexOf + 1);
    int _length = prefix.length();
    String _substring = prefix.substring(_plus, _length);
    boolean _equals = Objects.equal(_name_1, _substring);
    if (_equals) {
      int _lastIndexOf_1 = prefix.lastIndexOf(".");
      String _substring_1 = prefix.substring(0, _lastIndexOf_1);
      prefix = _substring_1;
    }
    String _xifexpression = null;
    Iterable<MdGroup> _groups = this.getGroups(member);
    boolean _isEmpty = IterableExtensions.isEmpty(_groups);
    if (_isEmpty) {
      _xifexpression = "";
    } else {
      _xifexpression = ".";
    }
    String _plus_1 = (prefix + _xifexpression);
    Iterable<MdGroup> _groups_1 = this.getGroups(member);
    final Function1<MdGroup, String> _function = new Function1<MdGroup, String>() {
      @Override
      public String apply(final MdGroup it) {
        return it.getName();
      }
    };
    Iterable<String> _map = IterableExtensions.<MdGroup, String>map(_groups_1, _function);
    String _join = IterableExtensions.join(_map, ".");
    String _plus_2 = (_plus_1 + _join);
    String _plus_3 = (_plus_2 + ".");
    String _name_2 = member.getName();
    return (_plus_3 + _name_2);
  }
  
  private String getConstantName(final MdBundleMember member) {
    final String name = member.getName();
    if ((name != null)) {
      final StringBuilder result = new StringBuilder();
      Iterable<MdGroup> _groups = this.getGroups(member);
      final Function1<MdGroup, String> _function = new Function1<MdGroup, String>() {
        @Override
        public String apply(final MdGroup it) {
          String _name = it.getName();
          return MetaDataJvmModelInferrer.this.toUpperCaseWithUnderscores(_name);
        }
      };
      Iterable<String> _map = IterableExtensions.<MdGroup, String>map(_groups, _function);
      String _join = IterableExtensions.join(_map, "_");
      result.append(_join);
      int _length = result.length();
      boolean _greaterThan = (_length > 0);
      if (_greaterThan) {
        result.append("_");
      }
      String _upperCaseWithUnderscores = this.toUpperCaseWithUnderscores(name);
      result.append(_upperCaseWithUnderscores);
      return result.toString();
    }
    return null;
  }
  
  private String toUpperCaseWithUnderscores(final String str) {
    final StringBuilder result = new StringBuilder();
    for (int i = 0; (i < str.length()); i++) {
      {
        final char c = str.charAt(i);
        if ((Character.isUpperCase(c) && (i > 0))) {
          result.append("_");
        }
        char _upperCase = Character.toUpperCase(c);
        result.append(_upperCase);
      }
    }
    return result.toString();
  }
  
  private String getDefaultConstantName(final MdOption option) {
    String _constantName = this.getConstantName(option);
    return (_constantName + "_DEFAULT");
  }
  
  private String getLowerBoundConstantName(final MdOption option) {
    String _constantName = this.getConstantName(option);
    return (_constantName + "_LOWER_BOUND");
  }
  
  private String getUpperBoundConstantName(final MdOption option) {
    String _constantName = this.getConstantName(option);
    return (_constantName + "_UPPER_BOUND");
  }
  
  private String getDependencyConstantName(final MdOptionDependency dependency) {
    String _xblockexpression = null;
    {
      EObject _eContainer = dependency.eContainer();
      final MdOption option = ((MdOption) _eContainer);
      String _constantName = this.getConstantName(option);
      String _plus = (_constantName + "_DEP_");
      MdOption _target = dependency.getTarget();
      String _constantName_1 = this.getConstantName(_target);
      _xblockexpression = (_plus + _constantName_1);
    }
    return _xblockexpression;
  }
  
  private String toCodeString(final String s) {
    if ((s == null)) {
      return "null";
    } else {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("\"");
      String _convertToJavaString = Strings.convertToJavaString(s);
      _builder.append(_convertToJavaString, "");
      _builder.append("\"");
      return _builder.toString();
    }
  }
  
  private String shrinkWhiteSpace(final String s) {
    if ((s == null)) {
      return null;
    }
    final StringBuilder result = new StringBuilder();
    boolean shrink = true;
    for (int i = 0; (i < s.length()); i++) {
      {
        final char c = s.charAt(i);
        boolean _isWhitespace = Character.isWhitespace(c);
        if (_isWhitespace) {
          if ((!shrink)) {
            result.append(" ");
          }
          shrink = true;
        } else {
          result.append(c);
          shrink = false;
        }
      }
    }
    if (((result.length() > 0) && Character.isWhitespace(result.charAt((result.length() - 1))))) {
      int _length = result.length();
      int _minus = (_length - 1);
      result.deleteCharAt(_minus);
    }
    return result.toString();
  }
  
  private String trimLines(final String s) {
    String _xifexpression = null;
    if ((s == null)) {
      _xifexpression = null;
    } else {
      String[] _split = s.split("\r?\n");
      final Function1<String, String> _function = new Function1<String, String>() {
        @Override
        public String apply(final String it) {
          return it.trim();
        }
      };
      List<String> _map = ListExtensions.<String, String>map(((List<String>)Conversions.doWrapArray(_split)), _function);
      _xifexpression = IterableExtensions.join(_map, "\n");
    }
    return _xifexpression;
  }
  
  public void infer(final EObject model, final IJvmDeclaredTypeAcceptor acceptor, final boolean isPreIndexingPhase) {
    if (model instanceof MdModel) {
      _infer((MdModel)model, acceptor, isPreIndexingPhase);
      return;
    } else if (model != null) {
      _infer(model, acceptor, isPreIndexingPhase);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(model, acceptor, isPreIndexingPhase).toString());
    }
  }
}
