import { decode, encode } from 'sourcemap-codec';
import resolveUri from '@jridgewell/resolve-uri';

function resolve(input, base) {
    // The base is always treated as a directory, if it's not empty.
    // https://github.com/mozilla/source-map/blob/8cb3ee57/lib/util.js#L327
    // https://github.com/chromium/chromium/blob/da4adbb3/third_party/blink/renderer/devtools/front_end/sdk/SourceMap.js#L400-L401
    if (base && !base.endsWith('/'))
        base += '/';
    return resolveUri(input, base);
}

/**
 * Removes everything after the last "/", but leaves the slash.
 */
function stripFilename(path) {
    if (!path)
        return '';
    const index = path.lastIndexOf('/');
    return path.slice(0, index + 1);
}

function maybeSort(mappings, owned) {
    const unsortedIndex = nextUnsortedSegmentLine(mappings, 0);
    if (unsortedIndex === mappings.length)
        return mappings;
    // If we own the array (meaning we parsed it from JSON), then we're free to directly mutate it. If
    // not, we do not want to modify the consumer's input array.
    if (!owned)
        mappings = mappings.slice();
    for (let i = unsortedIndex; i < mappings.length; i = nextUnsortedSegmentLine(mappings, i + 1)) {
        mappings[i] = sortSegments(mappings[i], owned);
    }
    return mappings;
}
function nextUnsortedSegmentLine(mappings, start) {
    for (let i = start; i < mappings.length; i++) {
        if (!isSorted(mappings[i]))
            return i;
    }
    return mappings.length;
}
function isSorted(line) {
    for (let j = 1; j < line.length; j++) {
        if (line[j][0] < line[j - 1][0]) {
            return false;
        }
    }
    return true;
}
function sortSegments(line, owned) {
    if (!owned)
        line = line.slice();
    return line.sort(sortComparator);
}
function sortComparator(a, b) {
    return a[0] - b[0];
}

/**
 * A binary search implementation that returns the index if a match is found.
 * If no match is found, then the left-index (the index associated with the item that comes just
 * before the desired index) is returned. To maintain proper sort order, a splice would happen at
 * the next index:
 *
 * The `comparator` callback receives both the `item` under comparison and the
 * needle we are searching for. It must return `0` if the `item` is a match,
 * any negative number if `item` is too small (and we must search after it), or
 * any positive number if the `item` is too large (and we must search before
 * it).
 *
 *
 * ```js
 * const array = [1, 3];
 * const needle = 2;
 * const index = binarySearch(array, needle, (item, needle) => item - needle);
 *
 * assert.equal(index, 0);
 * array.splice(index + 1, 0, needle);
 * assert.deepEqual(array, [1, 2, 3]);
 * ```
 */
function binarySearch(haystack, needle, low, high) {
    while (low <= high) {
        const mid = low + ((high - low) >> 1);
        const cmp = haystack[mid][0] - needle;
        if (cmp === 0) {
            return mid;
        }
        if (cmp < 0) {
            low = mid + 1;
        }
        else {
            high = mid - 1;
        }
    }
    return low - 1;
}
function memoizedState() {
    return {
        _lastLine: -1,
        _lastColumn: -1,
        _lastIndex: -1,
    };
}
/**
 * This overly complicated beast is just to record the last tested line/column and the resulting
 * index, allowing us to skip a few tests if mappings are monotonically increasing.
 */
function memoizedBinarySearch(haystack, needle, state, line, column) {
    const { _lastLine: lastLine, _lastColumn: lastColumn, _lastIndex: lastIndex } = state;
    let low = 0;
    let high = haystack.length - 1;
    if (line === lastLine) {
        if (column === lastColumn) {
            return lastIndex;
        }
        if (column >= lastColumn) {
            low = Math.max(lastIndex, 0);
        }
        else {
            high = lastIndex;
        }
    }
    state._lastLine = line;
    state._lastColumn = column;
    return (state._lastIndex = binarySearch(haystack, needle, low, high));
}

const INVALID_MAPPING = Object.freeze({
    source: null,
    line: null,
    column: null,
    name: null,
});
class TraceMap {
    constructor(map, mapUrl) {
        this._binarySearchMemo = memoizedState();
        const isString = typeof map === 'string';
        const parsed = isString ? JSON.parse(map) : map;
        const { version, file, names, sourceRoot, sources, sourcesContent } = parsed;
        this.version = version;
        this.file = file;
        this.names = names;
        this.sourceRoot = sourceRoot;
        this.sources = sources;
        this.sourcesContent = sourcesContent;
        const from = resolve(sourceRoot || '', stripFilename(mapUrl));
        this.resolvedSources = sources.map((s) => resolve(s || '', from));
        const { mappings } = parsed;
        if (typeof mappings === 'string') {
            this._encoded = mappings;
            this._decoded = maybeSort(decode(mappings), true);
        }
        else {
            this._encoded = undefined;
            this._decoded = maybeSort(mappings, isString);
        }
    }
    /**
     * Returns the encoded (VLQ string) form of the SourceMap's mappings field.
     */
    encodedMappings() {
        var _a;
        return ((_a = this._encoded) !== null && _a !== void 0 ? _a : (this._encoded = encode(this._decoded)));
    }
    /**
     * Returns the decoded (array of lines of segments) form of the SourceMap's mappings field.
     */
    decodedMappings() {
        return this._decoded;
    }
    /**
     * Similar to Array.p.map, maps each segment into a new  segment. Passes -1 for any values that do
     * not exist in the SourceMapSegment. Both generatedLine and generatedColumn are 0-based.
     */
    map(fn) {
        const mapOut = [];
        const decoded = this._decoded;
        for (let i = 0; i < decoded.length; i++) {
            const line = decoded[i];
            const lineOut = [];
            mapOut.push(lineOut);
            for (let j = 0; j < line.length; j++) {
                const seg = line[j];
                const { length } = seg;
                let segOut;
                if (length === 4)
                    segOut = fn(i, seg[0], seg[1], seg[2], seg[3], -1);
                else if (length === 5)
                    segOut = fn(i, seg[0], seg[1], seg[2], seg[3], seg[4]);
                else
                    segOut = fn(i, seg[0], -1, -1, -1, -1);
                if (segOut != null)
                    lineOut.push(segOut);
            }
        }
        return mapOut;
    }
    /**
     * A low-level API to find the segment associated with a generated line/column (think, from a
     * stack trace). Line and column here are 0-based, unlike `originalPositionFor`.
     */
    traceSegment(line, column) {
        const decoded = this._decoded;
        // It's common for parent source maps to have pointers to lines that have no
        // mapping (like a "//# sourceMappingURL=") at the end of the child file.
        if (line >= decoded.length)
            return null;
        const segments = decoded[line];
        const index = memoizedBinarySearch(segments, column, this._binarySearchMemo, line, column);
        // we come before any mapped segment
        if (index < 0)
            return null;
        return segments[index];
    }
    /**
     * A higher-level API to find the source/line/column associated with a generated line/column
     * (think, from a stack trace). Line is 1-based, but column is 0-based, due to legacy behavior in
     * `source-map` library.
     */
    originalPositionFor({ line, column }) {
        if (line < 1)
            throw new Error('`line` must be greater than 0 (lines start at line 1)');
        if (column < 0) {
            throw new Error('`column` must be greater than or equal to 0 (columns start at column 0)');
        }
        const segment = this.traceSegment(line - 1, column);
        if (segment == null)
            return INVALID_MAPPING;
        if (segment.length == 1)
            return INVALID_MAPPING;
        const { names, resolvedSources } = this;
        return {
            source: resolvedSources[segment[1]],
            line: segment[2] + 1,
            column: segment[3],
            name: segment.length === 5 ? names[segment[4]] : null,
        };
    }
}

export { TraceMap, TraceMap as default };
//# sourceMappingURL=trace-mapping.mjs.map
