/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.rdf4j.sail.nativerdf.datastore;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.NoSuchElementException;
import org.eclipse.rdf4j.common.io.NioFile;
import org.eclipse.rdf4j.sail.nativerdf.NativeStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DataFile
implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(DataFile.class);
    private static final byte[] MAGIC_NUMBER = new byte[]{110, 100, 102};
    private static final byte FILE_FORMAT_VERSION = 1;
    private static final long HEADER_LENGTH = MAGIC_NUMBER.length + 1;
    private final NioFile nioFile;
    private final boolean forceSync;
    private volatile long nioFileSize;
    private final ByteBuffer buffer = ByteBuffer.allocate(4096);
    int dataLengthApproximateAverage = 25;

    public DataFile(File file) throws IOException {
        this(file, false);
    }

    public DataFile(File file, boolean forceSync) throws IOException {
        this.nioFile = new NioFile(file);
        this.forceSync = forceSync;
        try {
            if (this.nioFile.size() == 0L) {
                this.nioFile.writeBytes(MAGIC_NUMBER, 0L);
                this.nioFile.writeByte((byte)1, MAGIC_NUMBER.length);
                this.sync();
            } else {
                if (this.nioFile.size() < HEADER_LENGTH) {
                    throw new IOException("File too small to be a compatible data file");
                }
                if (!Arrays.equals(MAGIC_NUMBER, this.nioFile.readBytes(0L, MAGIC_NUMBER.length))) {
                    throw new IOException("File doesn't contain compatible data records");
                }
                byte version = this.nioFile.readByte(MAGIC_NUMBER.length);
                if (version > 1) {
                    throw new IOException("Unable to read data file; it uses a newer file format");
                }
                if (version != 1) {
                    throw new IOException("Unable to read data file; invalid file format version: " + version);
                }
            }
        }
        catch (IOException e) {
            this.nioFile.close();
            throw e;
        }
        this.nioFileSize = this.nioFile.size();
    }

    public File getFile() {
        return this.nioFile.getFile();
    }

    public long storeData(byte[] data) throws IOException {
        assert (data != null) : "data must not be null";
        long offset = this.nioFileSize;
        if (data.length + 4 > this.buffer.capacity()) {
            this.flush();
            ByteBuffer buf = ByteBuffer.allocate(data.length + 4);
            buf.putInt(data.length);
            buf.put(data);
            buf.rewind();
            this.nioFile.write(buf, offset);
            this.nioFileSize += (long)buf.array().length;
        } else {
            if (data.length + 4 > this.remainingBufferCapacity()) {
                this.flush();
            }
            this.buffer.putInt(data.length);
            this.buffer.put(data);
            this.nioFileSize += (long)(data.length + 4);
        }
        return offset;
    }

    private synchronized void flush() throws IOException {
        int position = this.buffer.position();
        if (position == 0) {
            return;
        }
        this.buffer.position(0);
        byte[] byteToWrite = new byte[position];
        this.buffer.get(byteToWrite, 0, position);
        this.nioFile.write(ByteBuffer.wrap(byteToWrite), this.nioFileSize - (long)byteToWrite.length);
        this.buffer.position(0);
    }

    private int remainingBufferCapacity() {
        return this.buffer.capacity() - this.buffer.position();
    }

    public byte[] getData(long offset) throws IOException {
        assert (offset > 0L) : "offset must be larger than 0, is: " + offset;
        this.flush();
        byte[] data = new byte[this.dataLengthApproximateAverage * 2 + 4];
        ByteBuffer buf = ByteBuffer.wrap(data);
        this.nioFile.read(buf, offset);
        int dataLength = data[0] << 24 & 0xFF000000 | data[1] << 16 & 0xFF0000 | data[2] << 8 & 0xFF00 | data[3] & 0xFF;
        if (dataLength > 0x8000000 && NativeStore.SOFT_FAIL_ON_CORRUPT_DATA_AND_REPAIR_INDEXES) {
            logger.error("Data length is {}MB which is larger than 750MB. This is likely data corruption. Truncating length to 32 MB.", (Object)(dataLength / 0x100000));
            dataLength = 0x2000000;
        }
        try {
            if (dataLength <= data.length - 4) {
                this.dataLengthApproximateAverage = (int)Math.min(200.0, (double)this.dataLengthApproximateAverage / 100.0 * 99.0 + (double)dataLength / 100.0);
                return Arrays.copyOfRange(data, 4, dataLength + 4);
            }
            this.dataLengthApproximateAverage = Math.min(200, (this.dataLengthApproximateAverage + dataLength) / 2);
            data = new byte[dataLength];
            buf = ByteBuffer.wrap(data);
            this.nioFile.read(buf, offset + 4L);
            return data;
        }
        catch (OutOfMemoryError e) {
            if (dataLength > 0x8000000) {
                logger.error("Trying to read large amounts of data may be a sign of data corruption. Consider setting the system property org.eclipse.rdf4j.sail.nativerdf.softFailOnCorruptDataAndRepairIndexes to true");
            }
            throw e;
        }
    }

    public void clear() throws IOException {
        this.nioFile.truncate(HEADER_LENGTH);
        this.nioFileSize = HEADER_LENGTH;
        this.buffer.clear();
    }

    public void sync() throws IOException {
        this.flush();
        if (this.forceSync) {
            this.nioFile.force(false);
        }
    }

    public void sync(boolean force) throws IOException {
        this.flush();
        this.nioFile.force(force);
    }

    @Override
    public void close() throws IOException {
        this.flush();
        this.nioFile.close();
    }

    public DataIterator iterator() {
        try {
            this.flush();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        return new DataIterator();
    }

    public class DataIterator {
        private long position = HEADER_LENGTH;

        public boolean hasNext() {
            return this.position < DataFile.this.nioFileSize;
        }

        public byte[] next() throws IOException {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            byte[] data = DataFile.this.getData(this.position);
            this.position += (long)(4 + data.length);
            return data;
        }
    }
}

