/*******************************************************************************
 * Copyright (c) 2011-2014 Knowledge Computing Corp. and others
 *
 * This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *    Karl M. Davis (Knowledge Computing Corp.) - initial API and implementation
 *    Red Hat, Inc - refactoring and abstraction of the logic
 *******************************************************************************/

package org.eclipse.m2e.apt.internal;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jdt.apt.core.util.AptConfig;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;

import org.apache.maven.plugin.MojoExecution;

import org.eclipse.m2e.apt.MavenJdtAptPlugin;
import org.eclipse.m2e.apt.preferences.AnnotationProcessingMode;
import org.eclipse.m2e.apt.preferences.IPreferencesManager;
import org.eclipse.m2e.core.lifecyclemapping.model.IPluginExecutionMetadata;
import org.eclipse.m2e.core.project.IMavenProjectFacade;
import org.eclipse.m2e.core.project.configurator.AbstractBuildParticipant;
import org.eclipse.m2e.core.project.configurator.AbstractProjectConfigurator;
import org.eclipse.m2e.core.project.configurator.ProjectConfigurationRequest;
import org.eclipse.m2e.jdt.IClasspathDescriptor;
import org.eclipse.m2e.jdt.IJavaProjectConfigurator;


/**
 * <p>
 * This {@link AbstractProjectConfigurator} implementation will set the APT configuration for an Eclipse Java project.
 * </p>
 * <p>
 * Please note that the <code>maven-compiler-plugin</code> (at least as of version 2.3.2) will automatically perform
 * annotation processing and generate annotation sources. This processing will include all annotation processors in the
 * project's compilation classpath.
 * </p>
 * <p>
 * However, there are a couple of problems that prevent the <code>maven-compiler-plugin</code>'s annotation processing
 * from being sufficient when run within m2eclipse:
 * </p>
 * <ul>
 * <li>The generated annotation sources are not added to the Maven project's source folders (nor should they be) and are
 * thus not found by m2eclipse.</li>
 * <li>Due to contention between Eclipse's JDT compilation and <code>maven-compiler-plugin</code> compilation, the Java
 * compiler used by Eclipse may not recognize when the generated annotation sources/classes are out of date.</li>
 * </ul>
 * <p>
 * The {@link AbstractAptProjectConfigurator} works around those limitations by configuring Eclipse's built-in
 * annotation processing: APT. Unfortunately, the APT configuration will not allow for libraries, such as m2eclipse's
 * "Maven Dependencies" to be used in the search path for annotation processors. Instead, the
 * {@link AbstractAptProjectConfigurator} adds all of the project's <code>.jar</code> dependencies to the annotation
 * processor search path.
 * </p>
 */
public abstract class AbstractAptProjectConfigurator extends AbstractProjectConfigurator
    implements IJavaProjectConfigurator {

  protected abstract AptConfiguratorDelegate getDelegate(AnnotationProcessingMode mode);

  /**
   * {@inheritDoc}
   */
  @Override
  public void configure(ProjectConfigurationRequest request, IProgressMonitor monitor) throws CoreException {
    // This method may be called with null parameters to ensure its API is correct. We
    // can ignore such calls.

    if((request == null) || (monitor == null)) {
      return;
    }

    // Get the objects needed for APT configuration
    IMavenProjectFacade mavenProjectFacade = request.mavenProjectFacade();

    AnnotationProcessingMode mode = getAnnotationProcessorMode(mavenProjectFacade);

    mavenProjectFacade.createExecutionContext().execute((c, m) -> {
      AptConfiguratorDelegate configuratorDelegate = getDelegate(mode);
      configuratorDelegate.setSession(c.getSession());
      configuratorDelegate.setFacade(mavenProjectFacade);

      // Configure APT
      if(!configuratorDelegate.isIgnored(monitor)) {
        configuratorDelegate.configureProject(monitor);
      }
      return null;
    }, monitor);

    configureAptReconcile(mavenProjectFacade.getProject());
  }

  /**
   * reconcile is enabled by default while enabling apt for maven-compiler-plugin, As Annotation processing usually
   * takes a long time for even a java file change, and what's more, validate a jsp also triggers apt reconcile as jsp
   * compiles into java, this option is provided to switch off the "Processing on Edit" feature.
   *
   * @throws CoreException
   */
  private void configureAptReconcile(IProject project) throws CoreException {
    if(project.hasNature(JavaCore.NATURE_ID)) {
      IJavaProject jp = JavaCore.create(project);
      if((jp != null) && AptConfig.isEnabled(jp)) {
        boolean shouldEnable = MavenJdtAptPlugin.getPreferencesManager()
            .shouldEnableAnnotationProcessDuringReconcile(project);
        if(shouldEnable && !AptConfig.shouldProcessDuringReconcile(jp)) {
          AptConfig.setProcessDuringReconcile(jp, true);
        }
        if(!shouldEnable && AptConfig.shouldProcessDuringReconcile(jp)) {
          AptConfig.setProcessDuringReconcile(jp, false);
        }
      }
    }
  }

  @Override
  public void configureClasspath(IMavenProjectFacade facade, IClasspathDescriptor classpath, IProgressMonitor monitor) {
    /*
     * Implementations of this method are supposed to configure the Maven project
     * classpath: the "Maven Dependencies" container. We don't have any need to do
     * that here.
     */
  }

  @Override
  public void configureRawClasspath(ProjectConfigurationRequest request, IClasspathDescriptor classpath,
      IProgressMonitor monitor) throws CoreException {
    /*
     * We need to prevent/recover from the JavaProjectConfigurator removing the
     * generated annotation sources directory from the classpath: it will be added
     * when we configure the Eclipse APT preferences and then removed when the
     * JavaProjectConfigurator runs.
     */
    // Get the various project references we'll need
    IProject eclipseProject = request.mavenProjectFacade().getProject();
    if(!eclipseProject.hasNature(JavaCore.NATURE_ID)) {
      return;
    }

    request.mavenProjectFacade().createExecutionContext().execute((c, m) -> {
      AptConfiguratorDelegate delegate = getDelegate(request.mavenProjectFacade());
      delegate.setFacade(request.mavenProjectFacade());
      delegate.setSession(c.getSession());
      // If this isn't a Java project, we have nothing to do
      if(!delegate.isIgnored(monitor)) {
        delegate.configureClasspath(classpath, monitor);
      }
      return null;
    }, monitor);
  }

  @Override
  public AbstractBuildParticipant getBuildParticipant(IMavenProjectFacade projectFacade, MojoExecution execution,
      IPluginExecutionMetadata executionMetadata) {

    AptConfiguratorDelegate configuratorDelegate;
    configuratorDelegate = getDelegate(projectFacade);
    return configuratorDelegate.getMojoExecutionBuildParticipant(execution);
  }

  private AptConfiguratorDelegate getDelegate(IMavenProjectFacade facade) {
    AnnotationProcessingMode mode = getAnnotationProcessorMode(facade);
    return getDelegate(mode);
  }

  private AnnotationProcessingMode getAnnotationProcessorMode(IMavenProjectFacade facade) {
    IPreferencesManager preferencesManager = MavenJdtAptPlugin.getPreferencesManager();
    return preferencesManager.getAnnotationProcessorMode(facade.getProject());
  }

}
