/*********************************************************************
 * Copyright (c) 2017 Boeing
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Boeing - initial API and implementation
 **********************************************************************/

package org.eclipse.osee.framework.ui.skynet.explorer;

import java.util.Collection;
import org.eclipse.jface.window.Window;
import org.eclipse.osee.framework.core.data.ArtifactId;
import org.eclipse.osee.framework.core.data.ArtifactToken;
import org.eclipse.osee.framework.core.data.BranchToken;
import org.eclipse.osee.framework.skynet.core.artifact.search.ArtifactQuery;
import org.eclipse.osee.framework.ui.skynet.branch.ViewApplicabilityUtil;
import org.eclipse.osee.framework.ui.skynet.widgets.dialog.ViewBranchViewFilterTreeDialog;
import org.eclipse.osee.framework.ui.swt.ALayout;
import org.eclipse.osee.framework.ui.swt.Displays;
import org.eclipse.osee.framework.ui.swt.Widgets;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.forms.widgets.FormText;

/**
 * @author Megumi Telles
 */
public class ArtifactExplorerViewApplicability {

   private final Composite parent;
   private FormText text;
   private Button button;
   private SelectionAdapter changeableAdapter;
   private final ArtifactExplorer explorer;

   public ArtifactExplorerViewApplicability(Composite parent, ArtifactExplorer explorer) {
      this.parent = parent;
      this.explorer = explorer;
   }

   public void create() {
      Composite applicabilityComp = new Composite(parent, SWT.WRAP);
      applicabilityComp.setLayout(ALayout.getZeroMarginLayout(2, false));
      applicabilityComp.setLayoutData(new GridData(SWT.BEGINNING, SWT.FILL, true, false));

      button = new Button(applicabilityComp, SWT.PUSH);
      button.setText("Set View");
      setButtonChangeable();

      text = new FormText(applicabilityComp, SWT.WRAP);
      GridData gd = new GridData(SWT.FILL, SWT.FILL, true, true);
      gd.widthHint = 200;
      text.setLayoutData(gd);
      text.setText(getArtifactViewApplicabiltyText(), true, false);
      text.setForeground(Displays.getSystemColor(SWT.COLOR_DARK_GRAY));

   }

   public void refresh() {
      text.setText(getArtifactViewApplicabiltyText(), true, false);
   }

   private void setButtonChangeable() {
      button.addSelectionListener(getChangeableAdapter());

   }

   private SelectionAdapter getChangeableAdapter() {
      if (changeableAdapter == null) {
         changeableAdapter = new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
               super.widgetSelected(e);
               changeView();
               refresh();
            }
         };
      }
      return changeableAdapter;
   }

   private String getArtifactViewApplicabiltyText() {
      String viewName = Widgets.NOT_SET;
      if (explorer != null) {
         BranchToken branch = explorer.getBranch();
         if (branch.isValid()) {
            if (!ViewApplicabilityUtil.isProductLineBranch(branch)) {
               button.setEnabled(false);
            } else {
               button.setEnabled(true);
               ArtifactId viewId = explorer.getViewId();
               if (viewId.isValid()) {
                  viewName = ArtifactQuery.getArtifactTokenFromId(explorer.getBranch(), viewId).getName();
               }
            }
         }
      }
      return String.format("<form><p>%s</p></form>", viewName);
   }

   private boolean changeView() {
      Collection<ArtifactToken> branchViews = ViewApplicabilityUtil.getBranchViewTokens(explorer.getBranch());
      ViewBranchViewFilterTreeDialog dialog =
         new ViewBranchViewFilterTreeDialog("Branch View", "Branch View", branchViews);
      dialog.setMultiSelect(false);
      int result = dialog.open();
      if (result == Window.OK) {
         explorer.setViewId(ArtifactId.valueOf(dialog.getSelection()));
         return true;
      }
      return false;
   }

}
