import { Protocol as Crdp } from 'devtools-protocol';
import { DebugProtocol } from 'vscode-debugprotocol';
import { BasePathTransformer, BaseSourceMapTransformer } from '..';
/**
 * Represents a reference to a source/script. `contents` is set if there are inlined sources.
 * Otherwise, scriptId can be used to retrieve the contents from the runtime.
 */
export interface ISourceContainer {
    /** The runtime-side scriptId of this script */
    scriptId?: Crdp.Runtime.ScriptId;
    /** The contents of this script, if they are inlined in the sourcemap */
    contents?: string;
    /** The authored path to this script (only set if the contents are inlined) */
    mappedPath?: string;
}
/**
 * A container class for loaded script files
 */
export declare class ScriptContainer {
    private _scriptsById;
    private _scriptsByUrl;
    private _sourceHandles;
    /**
     * @deprecated use the function calls instead
     */
    readonly scriptsByIdMap: Map<string, Crdp.Debugger.ScriptParsedEvent>;
    /**
     * Get a list of all currently loaded scripts
     */
    readonly loadedScripts: IterableIterator<Crdp.Debugger.ScriptParsedEvent>;
    /**
     * Get a script by its url
     */
    getScriptByUrl(url: string): Crdp.Debugger.ScriptParsedEvent;
    /**
     * Clear this container of all loaded scripts
     */
    reset(): void;
    /**
     * Add a newly parsed script to this container
     * @param script The scriptParsed event from the chrome-devtools target
     */
    add(script: Crdp.Debugger.ScriptParsedEvent): void;
    /**
     * Get a script by its chrome-devtools script identifier
     * @param id The script id which came from a chrome-devtools scriptParsed event
     */
    getScriptById(id: string): Crdp.Debugger.ScriptParsedEvent;
    /**
     * Get a list of all loaded script urls (as a string)
     */
    getAllScriptsString(pathTransformer: BasePathTransformer, sourceMapTransformer: BaseSourceMapTransformer): Promise<string>;
    /**
     * Get a script string?
     */
    getOneScriptString(runtimeScriptPath: string, pathTransformer: BasePathTransformer, sourceMapTransformer: BaseSourceMapTransformer): Promise<string>;
    /**
     * Get the existing handle for this script, identified by runtime scriptId, or create a new one
     */
    getSourceReferenceForScriptId(scriptId: Crdp.Runtime.ScriptId): number;
    /**
     * Get the existing handle for this script, identified by the on-disk path it was mapped to, or create a new one
     */
    getSourceReferenceForScriptPath(mappedPath: string, contents: string): number;
    /**
     * Map a chrome script to a DAP source
     * @param script The scriptParsed event object from chrome-devtools target
     * @param origin The origin of the script (node only)
     */
    scriptToSource(script: Crdp.Debugger.ScriptParsedEvent, origin: string, remoteAuthority?: string): Promise<DebugProtocol.Source>;
    /**
     * Get a source handle by it's reference number
     * @param ref Reference number of a source object
     */
    getSource(ref: number): ISourceContainer;
    fakeUrlForSourceReference(sourceReference: number): string;
    displayNameForSourceReference(sourceReference: number): string;
    displayNameForScriptId(scriptId: number | string): string;
    /**
     * Called when returning a stack trace, for the path for Sources that have a sourceReference, so consumers can
     * tweak it, since it's only for display.
     */
    realPathToDisplayPath(realPath: string): string;
    /**
     * Get the original path back from a displayPath created from `realPathToDisplayPath`
     */
    displayPathToRealPath(displayPath: string): string;
}
