import type { Mapping } from '@volar/source-map';
import type * as ts from 'typescript';
import type { LinkedCodeMap } from './linkedCodeMap';
export interface Mapper {
    mappings: Mapping<CodeInformation>[];
    toSourceRange(start: number, end: number, fallbackToAnyMatch: boolean, filter?: (data: CodeInformation) => boolean): Generator<readonly [number, number, Mapping<CodeInformation>, Mapping<CodeInformation>]>;
    toGeneratedRange(start: number, end: number, fallbackToAnyMatch: boolean, filter?: (data: CodeInformation) => boolean): Generator<readonly [number, number, Mapping<CodeInformation>, Mapping<CodeInformation>]>;
    toSourceLocation(generatedOffset: number, filter?: (data: CodeInformation) => boolean): Generator<readonly [number, Mapping<CodeInformation>]>;
    toGeneratedLocation(sourceOffset: number, filter?: (data: CodeInformation) => boolean): Generator<readonly [number, Mapping<CodeInformation>]>;
}
export type MapperFactory = (mappings: Mapping<CodeInformation>[]) => Mapper;
export interface Language<T = unknown> {
    mapperFactory: MapperFactory;
    plugins: LanguagePlugin<T>[];
    scripts: {
        get(id: T): SourceScript<T> | undefined;
        set(id: T, snapshot: ts.IScriptSnapshot, languageId?: string, plugins?: LanguagePlugin<T>[]): SourceScript<T> | undefined;
        delete(id: T): void;
        fromVirtualCode(virtualCode: VirtualCode): SourceScript<T>;
    };
    maps: {
        get(virtualCode: VirtualCode, sourceScript: SourceScript<T>): Mapper;
        forEach(virtualCode: VirtualCode): Generator<[sourceScript: SourceScript<T>, map: Mapper]>;
    };
    linkedCodeMaps: {
        get(virtualCode: VirtualCode): LinkedCodeMap | undefined;
    };
    typescript?: {
        configFileName: string | undefined;
        sys: ts.System & {
            version?: number;
            sync?(): Promise<number>;
        };
        languageServiceHost: ts.LanguageServiceHost;
        getExtraServiceScript(fileName: string): TypeScriptExtraServiceScript | undefined;
        asScriptId(fileName: string): T;
        asFileName(scriptId: T): string;
    };
}
export interface SourceScript<T = unknown> {
    id: T;
    languageId: string;
    snapshot: ts.IScriptSnapshot;
    targetIds: Set<T>;
    associatedIds: Set<T>;
    associatedOnly: boolean;
    isAssociationDirty?: boolean;
    generated?: {
        root: VirtualCode;
        languagePlugin: LanguagePlugin<T>;
        embeddedCodes: Map<string, VirtualCode>;
    };
}
export type CodeMapping = Mapping<CodeInformation>;
export interface VirtualCode {
    id: string;
    languageId: string;
    snapshot: ts.IScriptSnapshot;
    mappings: CodeMapping[];
    associatedScriptMappings?: Map<unknown, CodeMapping[]>;
    embeddedCodes?: VirtualCode[];
    linkedCodeMappings?: Mapping[];
}
export interface CodeInformation {
    /** virtual code is expected to support verification */
    verification?: boolean | {
        shouldReport?(): boolean;
    };
    /** virtual code is expected to support assisted completion */
    completion?: boolean | {
        isAdditional?: boolean;
        onlyImport?: boolean;
    };
    /** virtual code is expected correctly reflect semantic of the source code */
    semantic?: boolean | {
        shouldHighlight?(): boolean;
    };
    /** virtual code is expected correctly reflect reference relationships of the source code */
    navigation?: boolean | {
        shouldRename?(): boolean;
        resolveRenameNewName?(newName: string): string;
        resolveRenameEditText?(newText: string): string;
    };
    /** virtual code is expected correctly reflect the structural information of the source code */
    structure?: boolean;
    /** virtual code is expected correctly reflect the format information of the source code */
    format?: boolean;
}
export interface TypeScriptServiceScript {
    code: VirtualCode;
    extension: '.ts' | '.js' | '.mts' | '.mjs' | '.cjs' | '.cts' | '.d.ts' | string;
    scriptKind: ts.ScriptKind;
    /** See #188 */
    preventLeadingOffset?: boolean;
}
export interface TypeScriptExtraServiceScript extends TypeScriptServiceScript {
    fileName: string;
}
export interface LanguagePlugin<T = unknown, K extends VirtualCode = VirtualCode> {
    /**
     * For files that are not opened in the IDE, the language ID will not be synchronized to the language server, so a hook is needed to parse the language ID of files that are known extension but not opened in the IDE.
     */
    getLanguageId(scriptId: T): string | undefined;
    /**
     * Generate a virtual code.
     */
    createVirtualCode?(scriptId: T, languageId: string, snapshot: ts.IScriptSnapshot, ctx: CodegenContext<T>): K | undefined;
    /**
     * Incremental update a virtual code. If not provide, call createVirtualCode again.
     */
    updateVirtualCode?(scriptId: T, virtualCode: K, newSnapshot: ts.IScriptSnapshot, ctx: CodegenContext<T>): K | undefined;
    /**
     * Cleanup a virtual code.
     */
    disposeVirtualCode?(scriptId: T, virtualCode: K): void;
    /**
     * Some file types should not be parsed or processed as TypeScript files,
     * as they are used only as sources for generated files.
     *
     * This functionality is required only in TS plugin mode.
     */
    isAssociatedFileOnly?(scriptId: T, languageId: string): boolean;
    typescript?: TypeScriptGenericOptions<K> & TypeScriptNonTSPluginOptions<K>;
}
export interface CodegenContext<T = unknown> {
    getAssociatedScript(scriptId: T): SourceScript<T> | undefined;
}
/**
 * The following options available to all situations.
 */
interface TypeScriptGenericOptions<K> {
    extraFileExtensions: ts.FileExtensionInfo[];
    resolveHiddenExtensions?: boolean;
    getServiceScript(root: K): TypeScriptServiceScript | undefined;
}
/**
 * The following options will not be available in TS plugin.
 */
interface TypeScriptNonTSPluginOptions<K> {
    getExtraServiceScripts?(fileName: string, rootVirtualCode: K): TypeScriptExtraServiceScript[];
    resolveLanguageServiceHost?(host: ts.LanguageServiceHost): ts.LanguageServiceHost;
}
export {};
