##########################################################
####### Single Starter/Generator electrical system #######
####################    Syd Adams    #####################
###### Based on Curtis Olson's nasal electrical code #####
##########################################################
##########################################################
######## Modified by Clement DE L'HAMAIDE for P92 ########
##########################################################
######## Modified by D-ECHO for CH650 (10/2020)	  ########
##########################################################

# Based on information from http://www.aircraft-battery.com/search-by-your-aircraft/battery_detail/156 and http://www.zenithair.net/wp-content/uploads/stolch750/data/Draft%20POH%20STOL%20CH%20750-SLSA%20June%202009.pdf

var last_time = 0.0;
var OutPuts = props.globals.getNode("/systems/electrical/outputs",1);
var Volts = props.globals.getNode("/systems/electrical/volts",1);
var Amps = props.globals.getNode("/systems/electrical/amps",1);
var DIMMER = props.globals.getNode("/controls/lighting/instruments-norm",1);
var NORM = 0.0357;
var Battery={};
var Alternator={};
var load = 0.0;

var sw = props.globals.getNode("/controls/switches");
var sw_e = props.globals.getNode("/controls/electric");
var sw_l = props.globals.getNode("/controls/lighting");
var cb = props.globals.initNode("/controls/circuit-breakers");

var serviceable = {
	electric:	props.globals.getNode("systems/electrical/serviceable",1),
	comm:		props.globals.getNode("/instrumentation/comm/serviceable"),
	xpdr:		props.globals.getNode("instrumentation/transponder/serviceable"),
};
var switches = {
	bat:		props.globals.getNode("/controls/engines/engine/master-bat",1),
	alt:		props.globals.getNode("/controls/engines/engine/master-alt",1),
	radio_master:	sw_e.initNode("radio-master",	1,	"BOOL"),
	starter:	props.globals.getNode("/controls/engines/engine[0]/starter", 1),
	strobe:		sw_l.initNode("strobe",		0,	"BOOL"),
	nav:		sw_l.initNode("nav-lights",	0,	"BOOL"),
	landing:	sw_l.initNode("landing-light",	0,	"BOOL"),
	taxi:		sw_l.initNode("taxi-light",	0,	"BOOL"),
	boost:		props.globals.initNode("/controls/engines/engine/fuel-pump", 0, "BOOL"),
};
var circuit_breakers = {
	com:	cb.initNode("com",	1,	"BOOL"),
	nav:	cb.initNode("nav",	1,	"BOOL"),
	xpdr:	cb.initNode("xpdr",	1,	"BOOL"),
	audio:	cb.initNode("audio",	1,	"BOOL"),
	gps:	cb.initNode("gps",	1,	"BOOL"),
	efis:	cb.initNode("efis",	1,	"BOOL"),
	ens:	cb.initNode("ens",	1,	"BOOL"),
	trim:	cb.initNode("trim",	1,	"BOOL"),
	start:	cb.initNode("start",	1,	"BOOL"),
	field:	cb.initNode("field",	1,	"BOOL"),
	flap:	cb.initNode("flap",	1,	"BOOL"),
	stall:	cb.initNode("stall",	1,	"BOOL"),
	aux:	cb.initNode("aux",	1,	"BOOL"),
	alt:	cb.initNode("alt",	1,	"BOOL"),
	batt:	cb.initNode("batt",	1,	"BOOL"),
};

# Estimate electrical load:
# Landing Light: (based on https://aeroleds.com/products/sunspot-36lx-landing/): 45 Watts
# Taxi Light: (based on https://aeroleds.com/products/sunspot-36lx-taxi/): 45 Watts
# Navigation Lights and Strobe Lights based on https://aeroleds.com/wp-content/uploads/2018/02/0008-0006-Linear-Pulsar-NS-Series-Installation.pdf:
#	Navigation:	8.4 Watts (steady)			-> *2
#	Strobe:		70 Watts (during flash, 0.2 secs)	-> *2
#			11.2 Watts (average)			-> *2
# Fuel Pump: 50 Watts (complete guess)
var consumers_main = { 
	# consumer name:  [  power (watts),  switch node,   circuit breaker node ],
	landing_light:	[ 45, switches.landing, nil,	OutPuts.initNode("landing-light")	],
	taxi_light:	[ 45, switches.taxi,    nil,	OutPuts.initNode("taxi-light")	 	],
	strobe_lights:	[ 22.4, switches.strobe, nil,	OutPuts.initNode("strobe-lights")	],
	nav_lights:	[ 16.8, switches.nav,   nil,	OutPuts.initNode("nav-lights")		],
	fuel_pump:	[ 50.0, switches.boost, nil,	OutPuts.initNode("fuel-pump")		],
};

var strobe = aircraft.light.new("/sim/model/lights/strobe-lights", [0.2, 1.25], "/systems/electrical/outputs/strobe-lights");

#var battery = Battery.new(volts,amps,amp_hours,charge_percent,charge_amps);

Battery = {
	new : func {
		m = { parents : [Battery] };
		m.ideal_volts = arg[0];
		m.ideal_amps = arg[1];
		m.amp_hours = arg[2];
		m.charge_percent = arg[3];
		m.charge_amps = arg[4];
		return m;
	},
	apply_load : func {
		var amphrs_used = arg[0] * (arg[1] / 3600.0);
		var percent_used = amphrs_used / me.amp_hours;
		me.charge_percent -= percent_used;
		if ( me.charge_percent < 0.0 ) {
			me.charge_percent = 0.0;
		} elsif ( me.charge_percent > 1.0 ) {
			me.charge_percent = 1.0;
		}
		return me.amp_hours * me.charge_percent;
	},
	get_output_volts : func {
		var x = 1.0 - me.charge_percent;
		var tmp = -(3.0 * x - 1.0);
		var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
		return me.ideal_volts * factor;
	},
	get_output_amps : func {
		var x = 1.0 - me.charge_percent;
		var tmp = -(3.0 * x - 1.0);
		var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
		return me.ideal_amps * factor;
	},	
	recharge : func {
		me.charge_percent = 1.0;
	},
};

# var alternator = Alternator.new("rpm-source",rpm_threshold,volts,amps);

Alternator = {
	new : func {
		m = { parents : [Alternator] };
		m.rpm_source =  props.globals.getNode(arg[0],1);
		m.rpm_threshold = arg[1];
		m.ideal_volts = arg[2];
		m.ideal_amps = arg[3];
		return m;
	},
	
	apply_load : func( amps, dt) {
		var factor = me.rpm_source.getValue() / me.rpm_threshold;
		if ( factor > 1.0 ){
			factor = 1.0;
		}
		var available_amps = me.ideal_amps * factor;
		return available_amps - amps;
	},
	
	get_output_volts : func {
		var factor = me.rpm_source.getValue() / me.rpm_threshold;
		if ( factor > 1.0 ) {
			factor = 1.0;
		}
		return me.ideal_volts * factor;
	},
	
	get_output_amps : func {
		var factor = me.rpm_source.getValue() / me.rpm_threshold;
		if ( factor > 1.0 ) {
			factor = 1.0;
		}
		return me.ideal_amps * factor;
	}
};

var battery = Battery.new(12.0, 20.0, 22.0, 1.0, 12.0);
var alternator = Alternator.new("/engines/engine[0]/rpm", 250.0, 14.0, 40.0);

setlistener("/sim/signals/fdm-initialized", func {
	foreach(var a; props.globals.getNode("/systems/electrical/outputs").getChildren()){
		a.setDoubleValue(0);
	}
	foreach(var a; keys(switches) ){
		switches[a].setBoolValue(0);
	}
	foreach(var a; keys(circuit_breakers) ){
		circuit_breakers[a].setBoolValue(1);
	}
	props.globals.getNode("/engines/engine/amp-v",1).setDoubleValue(0);
	props.globals.getNode("/controls/engines/engine/master-alt",1).setBoolValue(0);
	props.globals.getNode("/controls/engines/engine/master-bat",1).setBoolValue(0);
	settimer(update_electrical,1);
	print("Electrical System ... OK");
});

var bus_volts = 0.0;

var update_virtual_bus = func(dt) {
	var AltVolts = alternator.get_output_volts();
	var AltAmps = alternator.get_output_amps();
	var BatVolts = battery.get_output_volts();
	var BatAmps = battery.get_output_amps();
	var power_source = nil;
	
	if( serviceable.electric.getBoolValue() ){					# Electrical System is serviceable
		if ( switches.alt.getBoolValue() and circuit_breakers.alt.getBoolValue() and (AltAmps > BatAmps)){
			bus_volts = AltVolts;
			power_source = "alternator";
			battery.apply_load(-battery.charge_amps, dt);			# Charge the battery
		}elsif ( switches.bat.getBoolValue() and circuit_breakers.batt.getBoolValue() ){
			bus_volts = BatVolts;
			power_source = "battery";
			battery.apply_load(load, dt);	# Load in ampere
		}else {
			bus_volts = 0.0;
		}
	} else {									# Electrical System not serviceable
		bus_volts = 0.0;
	}
	
	load = 0.0;
	load += electrical_bus(bus_volts);
	load += avionics_bus(bus_volts);
	
	props.globals.getNode("/engines/engine/amp-v",1).setValue(bus_volts);		# TODO what does this affect?
	
	var bus_amps = 0.0;
	
	if (bus_volts > 1.0){
		if (power_source == "battery"){
			bus_amps = BatAmps-load;
		} else {
			bus_amps = battery.charge_amps;
		}
	}
	
	Amps.setValue(bus_amps);
	Volts.setValue(bus_volts);
	return load;
}

var electrical_bus = func(bus_volts){
	load = 0.0;
	
	foreach( var key; keys( consumers_main ) ){
		if( consumers_main[key][1].getBoolValue() and ( consumers_main[key][2] == nil or consumers_main[key][2].getBoolValue() ) ){
			consumers_main[key][3].setDoubleValue( bus_volts );
			if( bus_volts != 0 ){
				load += ( consumers_main[key][0] / bus_volts );
			}
		} else {
			consumers_main[key][3].setDoubleValue( 0.0 );
		}
	}
	
	# Starter has about 900W power according to https://www.ulforum.de/ultraleicht/forum/2_technik-und-flugzeuge/2466_anlasser-rotax-912-uls/seite-4.html
	if ( switches.starter.getBoolValue() and circuit_breakers.start.getBoolValue() ){
		OutPuts.getNode("starter",1).setValue(bus_volts);
		if(bus_volts != 0){
			load += (900 / bus_volts);
		}
	} else {
		OutPuts.getNode("starter",1).setValue(0.0);
	}
	
	return load;
}

var avionics_bus = func(bv) {
	
	load = 0.0;
	
	if( switches.radio_master.getBoolValue() ){
		var bus_volts = bv;
		
		if ( serviceable.comm.getBoolValue() and circuit_breakers.com.getBoolValue() ){
			OutPuts.getNode("comm",1).setValue(bus_volts);
			load += 0.00015;
		} else {
			OutPuts.getNode("comm",1).setValue(0.0);
		}
		
		if ( serviceable.xpdr.getBoolValue() and circuit_breakers.xpdr.getBoolValue() ){
			OutPuts.getNode("transponder",1).setValue(bus_volts);
			load += 0.00015;
		} else {
			OutPuts.getNode("transponder",1).setValue(0.0);
		}
		
		if ( circuit_breakers.efis.getBoolValue() ){
			OutPuts.getNode("efis",1).setValue(bus_volts);
			load += 0.00015;
		} else {
			OutPuts.getNode("efis",1).setValue(0.0);
		}
	}else{
		var bus_volts = 0.0;
		
		OutPuts.getNode("comm",1).setValue(0.0);
		OutPuts.getNode("transponder",1).setValue(0.0);
		OutPuts.getNode("efis",1).setValue(0.0);
	}
	
	
	return load;
}

var update_electrical = func {
	var time = getprop("/sim/time/elapsed-sec");
	var dt = time - last_time;
	last_time = time;
	update_virtual_bus(dt);
	settimer(update_electrical, 0);
}
