# EXPORT : functions ending by export are called from xml
# CRON : functions ending by cron are called from timer
# SCHEDULE : functions ending by schedule are called from cron


# =============
# CREW CALLOUTS 
# =============

Voice = {};

Voice.new = func {
   var obj = { parents : [Voice,State.new(),System.new("/systems/voice")],

               autopilotsystem : nil,

               automata : VoiceAutomata.new(),
               callout : Callout.new(),
               checklist : Emergency.new("/systems/voice"),
               checklist2 : VoiceChecklist.new(),
               check : VoiceCheck.new(),
               initial : VoiceInitial.new(),
               next : VoiceNext.new(),
               which : VoiceWhich.new(),
               phase : Voicephase.new("/systems/voice"),
               sensor : Voicesensor.new("/systems/voice"),
               intelligence : VoiceAI.new(),
               crewvoice : Crewvoice.new(),

               ABSENTSEC : 15.0,                                 # less attention
               HOLDINGSEC : 5.0,

               timervoice : nil,
               rates : 0.0,                                      # variable time step

               vertical : ""
         };

   obj.init();

   return obj;
}

Voice.init = func {
   me.timervoice = maketimer(constant.HUMANSEC, me, me.schedule);
   me.timervoice.singleShot = 1;
   me.timervoice.simulatedTime = 1;
   me.timervoice.start();
}

Voice.set_relation = func( autopilot) {
   me.autopilotsystem = autopilot;
   
   me.automata.set_relation( me.phase, me.callout );
   me.checklist2.set_relation( me.checklist );
   me.check.set_relation( me.checklist, me.crewvoice );
   me.initial.set_relation( me.automata, me.checklist2 );
   me.next.set_relation( me.autopilotsystem, me.automata, me.callout, me.checklist, me.checklist2, me.crewvoice, me.phase, me.sensor );
   me.which.set_relation( me.automata, me.callout, me.checklist, me.checklist2, me.crewvoice, me.phase, me.sensor );
}

Voice.set_rates = func( steps ) {
   me.rates = steps;

   me.phase.set_rates( me.rates );
}

# disable at startup
Voice.startupexport = func {
   var disable = me.itself["root-ctrl"].getChild("disable").getValue();
   
   # disable voice at startup
   if( disable ) {
       me.set_service(constant.FALSE);
   }
}

Voice.crewtextexport = func {
   me.crewvoice.textexport();
}

# enable crew
Voice.serviceexport = func {
   if( !me.dependency["crew"].getChild("serviceable").getValue() ) {
       me.initial.nochecklistinit();
   }
}

# enable voice
Voice.enableexport = func {
   var disable = me.itself["root-ctrl"].getChild("disable").getValue();
   var serviceable = !disable;
       
   me.set_service(serviceable);
}

Voice.toggleexport = func {
   if( me.dependency["crew"].getChild("serviceable").getValue() ) {
       me.groundexport();
   }
}

Voice.set_service = func( serviceable ) {
   me.itself["root"].getChild("serviceable").setValue(serviceable);
   
   if( !serviceable ) {
       # remove button from menu
       me.dependency["crew-ctrl"].getNode("menu").getChild("voice").setValue(constant.FALSE);
   }
   
   var node = me.dependency["crew-ctrl"].getNode("status");
   
   if( serviceable ) {
       node.getChild("allways").setValue(constant.FALSE);
       node.getChild("notify").setValue(constant.TRUE);
       node.getChild("timeout").setValue(constant.FALSE);
       node.getChild("none").setValue(constant.FALSE);
   }
   else {
       node.getChild("allways").setValue(constant.FALSE);
       node.getChild("notify").setValue(constant.FALSE);
       node.getChild("timeout").setValue(constant.FALSE);
       node.getChild("none").setValue(constant.TRUE);
   }
}


# -------------------------------
# ground checklists not in flight
# -------------------------------
Voice.afterlandingexport = func {
   if( me.intelligence.has_crew() ) {
       var result = constant.FALSE;

       # abort takeoff
       if( me.callout.is_holding() ) {
           result = constant.TRUE;
       }

       # aborted takeoff before V1
       elsif( me.callout.is_takeoff() and !me.automata.get_v1() ) {
           result = constant.TRUE;
       }

       # after landing
       elsif( me.callout.is_taxiway() ) {
           result = constant.TRUE;
       }

       # abort taxi
       elsif( me.checklist.is_started() or me.checklist.is_runway() ) {
           result = constant.TRUE;
       }

       if( result ) {
           me.initial.afterlandinginit();
       }
   }
}

Voice.parkingexport = func {
   if( me.intelligence.has_crew() ) {
       if( me.callout.is_terminal() ) {
           me.initial.parkinginit();
       }
   }
}

Voice.stopoverexport = func {
   if( me.intelligence.has_crew() ) {
       # at FG start, default is holding without checklist.
       if( me.callout.is_gate() or ( me.callout.is_holding() or me.callout.is_takeoff() ) ) {
           me.checklist.set_startup();
           me.initial.stopoverinit();
       }
   }
}

Voice.externalexport = func {
   if( me.intelligence.has_crew() ) {
       if( me.callout.is_gate() ) {
           me.initial.externalinit();
       }
   }
}

Voice.preliminaryexport = func {
   if( me.intelligence.has_crew() ) {
       if( me.callout.is_gate() ) {
           me.initial.preliminaryinit();
       }
   }
}

Voice.cockpitexport = func {
   if( me.intelligence.has_crew() ) {
       if( me.callout.is_gate() ) {
           me.initial.cockpitinit();
       }
   }
}

Voice.beforestartexport = func {
   if( me.intelligence.has_crew() ) {
       if( me.callout.is_gate() ) {
           me.initial.beforestartinit();
       }
   }
}

Voice.enginestartexport = func {
   if( me.intelligence.has_crew() ) {
       if( me.callout.is_gate() ) {
           me.initial.enginestartinit();
       }
   }
}

Voice.pushbackexport = func {
   if( me.intelligence.has_crew() ) {
       if( me.callout.is_gate() ) {
           me.initial.pushbackinit();
       }
   }
}

Voice.afterstartexport = func {
   if( me.intelligence.has_crew() ) {
       if( me.checklist.is_started() ) {
           me.initial.afterstartinit();
       }
   }
}

Voice.taxiexport = func {
   if( me.intelligence.has_crew() ) {
       # at FG start, default is holding without checklist.
       if( me.checklist.is_started() or ( me.callout.is_holding() or me.callout.is_takeoff() ) ) {
           me.checklist.set_startup();
           me.initial.taxiinit();
       }
   }
}

Voice.beforetakeoffexport = func {
   if( me.intelligence.has_crew() ) {
       if( me.callout.is_holding() or me.callout.is_takeoff() ) {
           me.checklist.set_startup();
           me.initial.beforetakeoffinit();
       }
   }
}


# ------------------------------------------
# flight checklists can be trained on ground
# ------------------------------------------
Voice.aftertakeoffexport = func {
   if( me.intelligence.has_crew() ) {
       me.initial.aftertakeoffinit();
   }
}

Voice.climbexport = func {
   if( me.intelligence.has_crew() ) {
       me.initial.climbinit();
   }
}

Voice.transsonicexport = func {
   if( me.intelligence.has_crew() ) {
       me.initial.transsonicinit();
   }
}

Voice.descentexport = func {
   if( me.intelligence.has_crew() ) {
       me.initial.descentinit();
   }
}

Voice.approachexport = func {
   if( me.intelligence.has_crew() ) {
       me.initial.approachinit();
   }
}

Voice.beforelandingexport = func {
   if( me.intelligence.has_crew() ) {
       me.initial.beforelandinginit();
   }
}

Voice.groundexport = func {
   if( me.intelligence.has_AI() ) {
       var presets = me.dependency["crew-ctrl"].getChild("presets").getValue();

       if( presets == 0 ) {
           me.beforetakeoffexport();
       }

       elsif( presets == 1 ) {
           me.taxiexport();
       }

       elsif( presets == 2 ) {
           me.stopoverexport();
       }
   }
}


# --------------------
# emergency procedures
# --------------------
Voice.fourengineflameoutexport = func {
   if( me.intelligence.has_crew() ) {
       me.initial.nochecklistinit();
       me.initial.fourengineflameoutinit();
   }
}

Voice.fourengineflameoutmach1export = func {
   if( me.intelligence.has_crew() ) {
       me.initial.nochecklistinit();
       me.initial.fourengineflameoutmach1init();
   }
}


# ------------------
# no menu, crew only
# ------------------
Voice.nochecklistexport = func {
   me.initial.nochecklistinit();
}

Voice.noemergencyexport = func {
   me.initial.noemergencyinit();
}

Voice.startedexport = func {
   me.initial.startedinit();
}

Voice.runwayexport = func {
   me.initial.runwayinit();
}

Voice.cruiseclimbexport = func {
   me.initial.cruiseclimbinit();
}


# ------------------------
# to unlock checklists bug
# ------------------------
Voice.abortexport = func {
   me.initial.nochecklistinit();

   me.captainfeedback("abort");
}


# ----------------------
# to unlock callouts bug
# ----------------------
Voice.taxiwayexport = func {
   me.initial.taxiwayinit();
}

Voice.terminalexport = func {
   me.initial.terminalinit();
}

Voice.gateexport = func {
   me.initial.gateinit();
}

Voice.takeoffexport = func {
   me.initial.takeoffinit();
}

Voice.flightexport = func {
   me.initial.flightinit();
}

Voice.landingexport = func {
   me.initial.landinginit();
}


# ------------
# voice checks
# ------------
Voice.captainfeedback = func( action ) {
   if( !me.is_state() ) {
       me.check.captainfeedback( action );
   }
}

Voice.captaincheck = func( action ) {
   if( !me.is_state() ) {
       me.check.captain( action );
   }
}

Voice.pilotcheck = func( action, argument = "" ) {
   if( !me.is_state() ) {
       me.check.pilot( action, argument );
   }
}

Voice.engineercheck = func( action ) {
   if( !me.is_state() ) {
       me.check.engineer( action );
   }
}


Voice.schedule = func {
   if( me.itself["root"].getChild("serviceable").getValue() ) {
       me.set_rates( me.ABSENTSEC );

       # sensors
       me.vertical = me.dependency["autoflight"].getChild("vertical").getValue();

       me.phase.schedule();
       me.sensor.schedule();

       me.crewvoice.schedule();

       # decisions
       me.nextcallout();
       me.which.schedule( me.is_moving(), me.vertical );

       me.playvoices();
   }

   else {
       me.rates = me.ABSENTSEC;

       me.initial.nocalloutinit();
       me.initial.nochecklistinit();
       me.initial.noemergencyinit();
   }

   me.timervoice.restart(me.rates);
}

Voice.nextcallout = func {
   if( me.callout.is_landing() or me.checklist.is_beforelanding() ) {
       me.set_rates( constant.HUMANSEC );

       me.next.landing( me.rates );
   }
   elsif( me.callout.is_takeoff() or me.checklist.is_aftertakeoff() ) {
       me.set_rates( constant.HUMANSEC );

       me.next.takeoff( me.rates, me.vertical );
   }
   # not on taxi
   elsif( me.callout.is_holding() and !me.checklist.is_runway() ) {
       me.set_rates( me.HOLDINGSEC );

       me.next.holding();
   }
   elsif( me.callout.is_goaround() ) {
       me.set_rates( constant.HUMANSEC );

       me.next.goaround();
   }
   elsif( !me.phase.on_ground() ) {
       me.next.flight( me.rates, me.vertical );
   }
}

Voice.playvoices = func {
   if( me.crewvoice.willplay() ) {
       me.set_rates( constant.HUMANSEC );
   }

   me.crewvoice.playvoices( me.rates );
}


# ===========
# VOICE WHICH 
# ===========

VoiceWhich = {};

VoiceWhich.new = func {
   var obj = { parents : [VoiceWhich,System.new("/systems/voice")],
   
               callout : nil,
               checklist : nil,
               checklist2 : nil,
               crewvoice : nil,
               phase : nil,
               sensor : nil,
               
               initial : VoiceInitial.new(),
               intelligence : VoiceAI.new(),

               airport : "",
               runway : ""
         };

   obj.init();
   
   return obj;
}

VoiceWhich.init = func {
}

VoiceWhich.set_relation = func( automataarg, calloutarg, checklistarg, checklist2arg, crewvoicearg, phasearg, sensorarg ) {
   me.callout = calloutarg;
   me.checklist = checklistarg;
   me.checklist2 = checklist2arg;
   me.crewvoice = crewvoicearg;
   me.phase = phasearg;
   me.sensor = sensorarg;
   
   me.initial.set_relation( automataarg, me.checklist2 );
}

VoiceWhich.schedule = func( moving, vertical ) {
   me.whichcallout( moving, vertical );
   me.whichchecklist();
}

# the voice must work with and without virtual crew.
VoiceWhich.whichcallout = func( moving, vertical ) {
   # user is on ground
   if( !moving ) {
       me.userground();
   }

   # user has performed a go around
   elsif( vertical == "goaround" and ( me.callout.is_landing() or me.callout.is_goaround() ) ) {
       if( me.callout.is_landing() ) {
           me.initial.goaroundinit();
       }
   }

   # checklists required all crew members.
   elsif( !me.phase.on_ground() ) {
       me.userair();
   }
}

VoiceWhich.whichchecklist = func {
   if( me.intelligence.has_crew() ) {
       # AI triggers automatically checklists
       if( me.intelligence.has_AI() and !me.checklist.is_emergency() ) {

           # aircraft is climbing
           if( me.phase.is_climb_threshold() ) {
               me.crewclimb();
           }

           # aircraft is descending
           elsif( me.phase.is_descent_threshold() ) {
               me.crewdescent();
           }

           # aircraft cruise
           elsif( !me.phase.on_ground() ) {
               me.crewcruise();
           }
       }
   }

   else {
       me.initial.nochecklistinit();
   }
}

VoiceWhich.userground = func {
    var curairport = me.noinstrument["presets"].getChild("airport-id").getValue();
    var currunway = me.noinstrument["presets"].getChild("runway").getValue();

    
    # user has started all engines
    if( me.sensor.is_allengines() ) {
        if( me.callout.is_landing() ) {
            # when missed by callout
            if( me.phase.is_speed_below( 20 ) ) {
                me.initial.landingend();
            }
        }
        
        # taxi with all engines, without AI
        elsif( me.callout.is_taxiway() ) {
        }
        
        # taxi with all engines
        elsif( me.callout.is_terminal() ) {
        }
       
        else {
            if( !me.callout.is_taxiway() and !me.callout.is_holding() ) {
                me.initial.takeoffinit();
            }

            # user has relocated on ground
            if( !me.callout.is_takeoff() and !me.callout.is_holding() ) {
                # flight may trigger at FG startup !
                if( curairport != me.airport or currunway != me.runway ) {
                    me.initial.takeoffinit();
                }
            }

            # user has set parking brakes at runway threshold
            if( me.callout.is_takeoff() ) {
                if( me.dependency["gear-ctrl"].getChild("brake-parking-lever").getValue() ) {
                    me.initial.holdinginit();
                }
            }
        }
    }

    # user has stopped all engines
    elsif( me.sensor.is_noengines() ) {
        me.initial.gateinit();
    }

    # user has stopped inboard engines
    elsif( !me.dependency["engine"][constantaero.ENGINE2].getChild("running").getValue() and
           !me.dependency["engine"][constantaero.ENGINE3].getChild("running").getValue() ) {
        me.initial.terminalinit();
    }


    me.airport = curairport;
    me.runway = currunway;
}

VoiceWhich.userair = func {
    # aircraft is climbing
    if( me.phase.is_climb_threshold() and me.phase.is_agl_climb() ) {
        me.initial.flightinit();
    }

    # aircraft is descending
    elsif( me.phase.is_descent_threshold() ) {
        if( me.phase.is_altitude_approach() ) {
            me.initial.landinginit();
        }

        else {
            me.initial.flightinit();
        }
    }

    # aircraft is flying
    elsif( !me.callout.is_takeoff() ) {
        me.initial.flightinit();
    }
}

VoiceWhich.crewcruise = func {
    if( me.phase.is_altitude_cruise() ) {
        # waits for checklist end
        if( !me.checklist.is_transsonic() ) {
            me.initial.cruiseclimbinit();
        }
    }

    elsif( me.phase.is_mach_climb() ) {
        me.initial.flightinit();
    }

    elsif( !me.callout.is_takeoff() and !me.checklist.is_aftertakeoff() ) {
        me.initial.flightinit();
    }
}

VoiceWhich.crewclimb = func {
    if( me.phase.is_altitude_cruise() ) {
        # waits for checklist end
        if( !me.checklist.is_transsonic() ) {
            me.initial.cruiseclimbinit();
        }
    }

    # transsonic
    elsif( me.phase.is_mach_supersonic() ) {
        me.initial.transsonicinit();
    }

    # subsonic
    elsif( me.phase.is_mach_climb() ) {
        if( !me.checklist2.is_climb() ) {
            me.initial.climbinit();
        }
    }

    # starting climb
    elsif( !me.callout.is_takeoff() and !me.checklist.is_aftertakeoff() ) {
        me.initial.flightinit();
    }
}

VoiceWhich.crewdescent = func {
    # landing
    if( me.phase.is_agl_landing() ) {
        # impossible just after a takeoff, must climb enough
        if( !me.callout.is_takeoff() ) {
            if( !me.checklist2.is_beforelanding() ) {
                me.initial.beforelandinginit();
            }
        }
    }

    # approaching
    elsif( me.phase.is_altitude_approach() ) {
        if( !me.checklist2.is_approach() ) {
            me.initial.approachinit();
        }
    }

    # ending cruise
    elsif( me.phase.is_mach_supersonic() ) {
        me.initial.descentinit();
    }
}


# ==========
# VOICE NEXT 
# ==========

VoiceNext = {};

VoiceNext.new = func {
   var obj = { parents : [VoiceNext],
   
               checklist2 : nil,
               crewvoice : nil,
               
               initial : VoiceInitial.new(),
               step : VoiceStep.new(),
               allways : VoiceAllways.new()
         };

   obj.init();
   
   return obj;
}

VoiceNext.init = func {
}

VoiceNext.set_relation = func( autopilot, automataarg, calloutarg, checklistarg, checklist2arg, crewvoicearg, phasearg, sensorarg ) {
   me.checklist2 = checklist2arg;
   me.crewvoice = crewvoicearg;
   
   me.initial.set_relation( automataarg, me.checklist2 );
   me.step.set_relation( autopilot, automataarg, calloutarg, checklistarg, me.checklist2, me.crewvoice, phasearg, sensorarg );
   me.allways.set_relation( autopilot, automataarg, me.crewvoice, phasearg, sensorarg );    
}

VoiceNext.holding = func {
   me.step.holding();

   me.checklist2.sendchecklist();
}

VoiceNext.takeoff = func( ratesec, vertical ) {
   me.step.takeoffpilot();

   if( !me.crewvoice.is_asynchronous() ) {
       me.step.takeoffclimb();
   }

   if( !me.crewvoice.is_asynchronous() ) {
       me.allways.takeoff();
   }

   if( !me.crewvoice.is_asynchronous() ) {
       me.allways.flight( ratesec, vertical );
   }

   if( !me.crewvoice.is_asynchronous() ) {
       me.allways.check();
   }

   me.checklist2.sendchecklist();
}

VoiceNext.flight = func( ratesec, vertical ) {
   me.allways.flight( ratesec, vertical );

   if( !me.crewvoice.is_asynchronous() ) {
       me.allways.check();
   }

   me.checklist2.sendchecklist();
}

VoiceNext.landing = func( ratesec ) {
   me.step.landingengineer();

   if( !me.crewvoice.is_asynchronous() ) {
       me.step.landingpilot();
   }

   if( !me.crewvoice.is_asynchronous() ) {
       me.allways.landing( ratesec );
   }

   if( !me.crewvoice.is_asynchronous() ) {
       me.allways.check();
   }

   me.checklist2.sendchecklist();
}

VoiceNext.goaround = func {
   me.step.goaround();

   if( !me.crewvoice.is_asynchronous() ) {
       me.allways.takeoff();
   }

   if( !me.crewvoice.is_asynchronous() ) {
       me.allways.check();
   }

   me.checklist2.sendchecklist();
}


# =============
# VOICE INITIAL 
# =============

VoiceInitial = {};

VoiceInitial.new = func {
   var obj = { parents : [VoiceInitial],
   
               automata : nil,
               checklist2 : nil,
               
               intelligence : VoiceAI.new()
         };

   obj.init();
   
   return obj;
}

VoiceInitial.init = func {
}

VoiceInitial.set_relation = func( automataarg, checklist2arg ) {
   me.automata = automataarg;
   me.checklist2 = checklist2arg;
}

VoiceInitial.landingend = func {
   if( me.intelligence.has_AI() ) {
       me.afterlandinginit();
   }
   else {
       me.taxiwayinit();
   }
}

VoiceInitial.noemergencyinit = func {
   me.checklist2.noemergencyinit();
}

VoiceInitial.nochecklistinit = func {
   me.checklist2.nochecklistinit();
}

VoiceInitial.nocalloutinit = func {
   me.automata.nocalloutinit();
}

VoiceInitial.taxiwayinit = func {
   me.automata.taxiwayinit();
}

VoiceInitial.afterlandinginit = func {
   me.checklist2.afterlandinginit();
}

VoiceInitial.terminalinit = func {
   me.automata.terminalinit();
}

VoiceInitial.parkinginit = func {
   me.checklist2.parkinginit();
}

VoiceInitial.gateinit = func {
   me.automata.gateinit();
}

VoiceInitial.stopoverinit = func {
   me.checklist2.stopoverinit();
}

VoiceInitial.externalinit = func {
   me.checklist2.externalinit();
}

VoiceInitial.preliminaryinit = func {
   me.checklist2.preliminaryinit();
}

VoiceInitial.cockpitinit = func {
   me.checklist2.cockpitinit();
}

VoiceInitial.beforestartinit = func {
   me.checklist2.beforestartinit();
}

VoiceInitial.enginestartinit = func {
   me.checklist2.enginestartinit();
}

VoiceInitial.pushbackinit = func {
   me.checklist2.pushbackinit();
}

VoiceInitial.startedinit = func {
   me.checklist2.startedinit();
}

VoiceInitial.afterstartinit = func {
   me.checklist2.afterstartinit();
}

VoiceInitial.taxiinit = func {
   me.checklist2.taxiinit();
}

VoiceInitial.runwayinit = func {
   me.checklist2.runwayinit();
}

VoiceInitial.beforetakeoffinit = func {
   me.checklist2.beforetakeoffinit();
}

VoiceInitial.holdinginit = func {
   me.automata.holdinginit();
}

VoiceInitial.takeoffinit = func ( overwrite = 0 ) {
   me.automata.takeoffinit( overwrite );
}

VoiceInitial.aftertakeoffinit = func {
   me.checklist2.aftertakeoffinit();
}

VoiceInitial.flightinit = func {
   me.automata.flightinit();
}

VoiceInitial.climbinit = func {
   me.checklist2.climbinit();
}

VoiceInitial.transsonicinit = func {
   me.checklist2.transsonicinit();
}

VoiceInitial.cruiseclimbinit = func {
   me.checklist2.cruiseclimbinit();
}

VoiceInitial.descentinit = func {
   me.checklist2.descentinit();
}

VoiceInitial.approachinit = func {
   me.checklist2.approachinit();
}

VoiceInitial.beforelandinginit = func {
   me.checklist2.beforelandinginit();
}

VoiceInitial.landinginit = func {
   me.automata.landinginit();
}

VoiceInitial.goaroundinit = func {
   me.automata.goaroundinit();
}

VoiceInitial.fourengineflameoutinit = func {
   me.checklist2.fourengineflameoutinit();
}

VoiceInitial.fourengineflameoutmach1init = func {
   me.checklist2.fourengineflameoutmach1init();
}


# ==============
# VOICE DECISION 
# ==============

VoiceDecision = {};

VoiceDecision.new = func {
   var obj = { parents : [VoiceDecision,System.new("/systems/voice")],
   
               autopilotsystem : nil,

               automata : nil,
               crewvoice : nil,
               phase : nil,
               sensor : nil,

               AGLLEVELFT : { "2500ft" : 2500, "1000ft" : 1000, "800ft" : 800, "500ft" : 500, "400ft" : 400, "300ft" : 300,
                              "200ft" : 200, "100ft" : 100, "50ft" : 50, "40ft" : 40, "30ft" : 30, "20ft" : 20, "15ft" : 15 }
        };

   obj.init();
   
   return obj;
}

VoiceDecision.init = func {
}


# ==========
# VOICE STEP 
# ==========

VoiceStep = {};

VoiceStep.new = func {
   var obj = { parents : [VoiceStep,VoiceDecision.new()],
               
               call : VoiceCall.new(),
               initial : VoiceInitial.new(),
               intelligence : VoiceAI.new(),
 
               SPEEDLEVELKT : { "240kt" : 240, "100kt" : 100, "60kt" : 60 }
        };

   obj.init();
   
   return obj;
}

VoiceStep.init = func {
}

VoiceStep.set_relation = func( autopilot, automataarg, calloutarg, checklistarg, checklist2arg, crewvoicearg, phasearg, sensorarg ) {
   me.autopilotsystem = autopilot;

   me.automata = automataarg;
   me.crewvoice = crewvoicearg;
   me.phase = phasearg;
   me.sensor = sensorarg;
   
   me.call.set_relation( calloutarg, checklistarg, crewvoicearg );
   me.initial.set_relation( me.automata, checklist2arg );
}

VoiceStep.holding = func {
   if( me.automata.isStep2("holding") ) {
       if( !me.dependency["gear-ctrl"].getChild("brake-parking-lever").getValue() ) {
           if( me.dependency["captain-ctrl"].getChild("countdown").getValue() ) {
               me.automata.setStep2( me.call.captain( "brakes3" ) );
           }

           else {
               me.initial.takeoffinit();
           }
       }
   }

   elsif( me.automata.isStep2("brakes3") ) {
       me.automata.setStep2( me.call.captain( "brakes2" ) );
   }

   elsif( me.automata.isStep2("brakes2") ) {
       me.automata.setStep2( me.call.captain( "brakes1" ) );
   }

   elsif( me.automata.isStep2("brakes1") ) {
       me.automata.setStep2( me.call.captain( "brakes" ) );
   }
   else {
       me.initial.takeoffinit();
   }
}

VoiceStep.takeoffclimb = func {
   if( me.automata.isStep2("takeoff") ) {
       if( me.phase.is_climb_threshold() and me.phase.is_agl_liftoff() ) {
           me.automata.setStep2( me.crewvoice.stepmember( "liftoff", "pilot", "climb" ) );
           if( me.intelligence.has_AI() ) {
               me.initial.aftertakeoffinit();
           }
       }
   }
}

VoiceStep.takeoffpilot = func {
   if( me.automata.isStep("takeoff") ) {
       if( me.phase.is_speed_above( me.SPEEDLEVELKT["60kt"] ) ) {
           me.automata.setStep( me.call.pilot( "airspeed" ) );
       }
   }

   elsif( me.automata.isStep("airspeed") ) {
       if( me.phase.is_speed_above( me.SPEEDLEVELKT["100kt"] ) ) {
           me.automata.setStep( me.call.pilot( "100kt" ) );
           me.call.engineer( "100kt" );
       }
   }

   elsif( me.automata.isStep("100kt") ) {
       if( me.phase.is_speed_above( me.sensor.Vkt( constantaero.V1EMPTYKT,
                                                   constantaero.V1FULLKT ) ) ) {
           me.automata.setStep( me.call.pilot( "V1" ) );
           me.automata.set_v1( constant.TRUE );
       }
   }

   elsif( me.automata.isStep("V1") ) {
       if( me.phase.is_speed_above( me.sensor.Vkt( constantaero.VREMPTYKT,
                                                   constantaero.VRFULLKT ) ) ) {
           me.automata.setStep( me.call.pilot( "VR" ) );
       }
   }

   elsif( me.automata.isStep("VR") ) {
       if( me.phase.is_speed_above( me.sensor.Vkt( constantaero.V2EMPTYKT,
                                                   constantaero.V2FULLKT ) ) ) {
           me.automata.setStep( me.call.pilot( "V2" ) );
           me.automata.set_v2( constant.TRUE );
       }
   }

   elsif( me.automata.isStep("V2") ) {
       if( me.phase.is_speed_above( me.SPEEDLEVELKT["240kt"] ) ) {
           me.automata.setStep( me.call.pilot( "240kt" ) );
       }
   }

   # aborted takeoff
   if( !me.automata.isStep("takeoff") ) {
       if( me.phase.is_speed_below( 20 ) ) {
           me.initial.takeoffinit( constant.TRUE );
       }
   }
}

VoiceStep.landingpilot = func {
   if( me.automata.isStep2("landing") ) {
       if( me.dependency["nav"].getChild("in-range").getValue() ) {
           if( me.autopilotsystem.is_engaged() and
               me.dependency["autoflight"].getChild("heading").getValue() == "nav1-hold" ) {
               me.automata.setStep2( me.call.pilot( "beambar" ) );
           }
       }
   }
   elsif( me.automata.isStep2("beambar") ) {
       if( me.dependency["nav"].getChild("in-range").getValue() and
           me.dependency["nav"].getChild("has-gs").getValue() ) {
           if( me.autopilotsystem.is_engaged() and
               me.dependency["autoflight"].getChild("altitude").getValue() == "gs1-hold" ) {
               me.automata.setStep2( me.call.pilot( "glideslope" ) );
           }
       }
   }
   elsif( me.automata.isStep2("glideslope") ) {
       if( me.phase.is_speed_below( 100 ) ) {
           me.automata.setStep2( me.call.pilot( "100kt" ) );
       }
   }
   elsif( me.automata.isStep2("100kt") ) {
       if( me.phase.is_speed_below( 75 ) ) {
           me.automata.setStep2( me.call.pilot( "75kt" ) );
       }
   }
   elsif( me.automata.isStep2("75kt") ) {
       if( me.phase.is_speed_below( 40 ) ) {
           me.automata.setStep2( me.call.pilot( "40kt" ) );
       }
   }
   elsif( me.automata.isStep2("40kt") ) {
       if( me.phase.is_speed_below( 20 ) ) {
           me.automata.setStep2( me.call.pilot( "20kt" ) );
           # wake up AI
           me.initial.landingend();
       }
   }
}

VoiceStep.landingengineer = func {
   if( me.automata.isStep("landing") ) {
       if( me.phase.is_agl_below_level( me.AGLLEVELFT["2500ft"] ) ) {
           me.automata.setStep( me.call.engineer( "2500ft" ) );
       }
   }

   elsif( me.automata.isStep("2500ft") ) {
       if( me.phase.is_agl_below_level( me.AGLLEVELFT["1000ft"] ) ) {
           me.automata.setStep( me.call.engineer( "1000ft" ) );
       }
   }

   elsif( me.automata.isStep("1000ft") ) {
       if( me.phase.is_agl_below_level( me.AGLLEVELFT["800ft"] ) ) {
           me.automata.setStep( me.call.engineer( "800ft" ) );
       }
   }

   elsif( me.automata.isStep("800ft") ) {
       if( me.phase.is_agl_below_level( me.AGLLEVELFT["500ft"] ) ) {
           me.automata.setStep( me.call.engineer( "500ft" ) );
       }
   }

   elsif( me.automata.isStep("500ft") ) {
       if( me.phase.is_agl_below_level( me.AGLLEVELFT["400ft"] ) ) {
           me.automata.setStep( me.call.engineer( "400ft" ) );
       }
   }

   elsif( me.automata.isStep("400ft") ) {
       if( me.phase.is_agl_below_level( me.AGLLEVELFT["300ft"] ) ) {
           me.automata.setStep( me.call.engineer( "300ft" ) );
       }
   }

   elsif( me.automata.isStep("300ft") ) {
       if( me.phase.is_agl_below_level( me.AGLLEVELFT["200ft"] ) ) {
           me.automata.setStep( me.call.engineer( "200ft" ) );
       }
   }

   elsif( me.automata.isStep("200ft") ) {
       if( me.phase.is_agl_below_level( me.AGLLEVELFT["100ft"] ) ) {
           me.automata.setStep( me.call.engineer( "100ft" ) );
       }
   }

   elsif( me.automata.isStep("100ft") ) {
       me.landingtouchdown( me.AGLLEVELFT["50ft"] );
   }

   elsif( me.automata.isStep("50ft") ) {
       me.landingtouchdown( me.AGLLEVELFT["40ft"] );
   }

   elsif( me.automata.isStep("40ft") ) {
       me.landingtouchdown( me.AGLLEVELFT["30ft"] );
   }

   elsif( me.automata.isStep("30ft") ) {
       me.landingtouchdown( me.AGLLEVELFT["20ft"] );
   }

   elsif( me.automata.isStep("20ft") ) {
       if( me.phase.is_agl_below_level( me.AGLLEVELFT["15ft"] ) ) {
           me.automata.setStep( me.call.engineer( "15ft" ) );
       }
   }
}

# can be faster
VoiceStep.landingtouchdown = func( limitft ) {
   if( 15 <= limitft and me.phase.is_agl_below_level( me.AGLLEVELFT["15ft"] ) ) {
       me.automata.setStep( me.call.engineer( "15ft" ) );
   }
   elsif( 20 <= limitft and me.phase.is_agl_below_level( me.AGLLEVELFT["20ft"] ) ) {
       me.automata.setStep( me.call.engineer( "20ft" ) );
   }
   elsif( 30 <= limitft and me.phase.is_agl_below_level( me.AGLLEVELFT["30ft"] ) ) {
       me.automata.setStep( me.call.engineer( "30ft" ) );
   }
   elsif( 40 <= limitft and me.phase.is_agl_below_level( me.AGLLEVELFT["40ft"] ) ) {
       me.automata.setStep( me.call.engineer( "40ft" ) );
   }
   elsif( 50 <= limitft and me.phase.is_agl_below_level( me.AGLLEVELFT["50ft"] ) ) {
       me.automata.setStep( me.call.engineer( "50ft" ) );
   }
}

VoiceStep.goaround = func {
   if( me.automata.isStep("goaround") ) {
       if( me.phase.is_climb_visible() ) {
           me.automata.setStep( me.call.pilot( "positivclimb" ) );
           if( me.intelligence.has_AI() ) {
               me.initial.aftertakeoffinit();
           }

           me.initial.flightinit();
       }
   }
}


# =============
# VOICE ALLWAYS 
# =============

VoiceAllways = {};

VoiceAllways.new = func {
   var obj = { parents : [VoiceAllways,VoiceDecision.new()],

               MODIFYSEC : 15.0,                                 # to modify something
                              
               selectft : 0.0,
               delayselectftsec : 0,

               FLAREDEG : 12.5,

               altitudeselect : constant.FALSE,
               fueltransfert : constant.FALSE
         };

   obj.init();
   
   return obj;
}

VoiceAllways.init = func {
   me.selectft = me.dependency["autoflight"].getChild("altitude-select").getValue();   
}

VoiceAllways.set_relation = func( autopilot, automataarg, crewvoicearg, phasearg, sensorarg ) {
   me.autopilotsystem = autopilot;

   me.automata = automataarg;
   me.crewvoice = crewvoicearg;
   me.phase = phasearg;
   me.sensor = sensorarg;
}

VoiceAllways.takeoff = func {
   if( !me.phase.on_ground() ) {
       if( me.phase.is_climb_decay() ) {
           me.crewvoice.stepmember( "negativvsi", "allways", "takeoff", constant.TRUE );
       }

       elsif( me.phase.is_speed_approach() and
              (  me.phase.is_climb_decrease() or
                 ( me.automata.get_v2() and
                   me.phase.is_speed_below( me.sensor.Vkt( constantaero.V2EMPTYKT,
                                                           constantaero.V2FULLKT ) ) ) ) ) {
           me.crewvoice.stepmember( "airspeeddecay", "allways", "takeoff", constant.TRUE );
       }
   }
}

VoiceAllways.flight = func( rates, vertical ) {
   if( !me.dependency["crew"].getChild("unexpected").getValue() ) {
       var altitudeft = me.dependency["autoflight"].getChild("altitude-select").getValue();
       if( me.selectft != altitudeft ) {
           me.selectft = altitudeft;
           me.delayselectftsec = rates;
       }
       elsif( me.delayselectftsec > 0 ) {
           if( me.delayselectftsec >= me.MODIFYSEC ) {
               if( me.crewvoice.stepmember( "altitudeset", "allways", "flight" ) ) {
                   me.delayselectftsec = 0;
               }
           }
           else {
               me.delayselectftsec = me.delayselectftsec + rates;
           }
       }


       if( !me.altitudeselect ) { 
           if( vertical == "altitude-acquire" ) {
               if( me.autopilotsystem.is_engaged() and
                   !me.autopilotsystem.altitudelight_on( me.phase.get_altitudeft(), me.selectft ) ) {
                   me.altitudeselect = constant.TRUE;
               }
           }
       }
       else { 
           if( me.autopilotsystem.is_engaged() and
               me.autopilotsystem.altitudelight_on( me.phase.get_altitudeft(), me.selectft ) ) {
               if( me.crewvoice.stepmember( "1000fttogo", "allways", "flight" ) ) {
                   me.altitudeselect = constant.FALSE;
               }
           }       
       }


       if( me.phase.is_altitude_level() ) {
           me.crewvoice.stepmember( "altimetercheck", "allways", "flight" );

           if( me.dependency["engineer"].getNode("cg/forward").getValue() ) {
               me.fueltransfert = constant.TRUE;
               me.crewvoice.stepmember( "cgforward", "engineer", "flight" );
           }
           elsif( me.dependency["engineer"].getNode("cg/aft").getValue() ) {
               me.fueltransfert = constant.TRUE;
               me.crewvoice.stepmember( "cgaft", "engineer", "flight" );
           }
           else {
               me.fueltransfert = constant.FALSE;
               me.crewvoice.stepmember( "cgcorrect", "engineer", "flight" );
           }
       }
       elsif( me.phase.is_altitude_transition() ) {
           me.crewvoice.stepmember( "transition", "allways", "flight" );
       }

       # fuel transfert is completed :
       # - climb at 26000 ft.
       # - cruise above 50000 ft.
       # - descent to 38000 ft.
       # - approach to 10000 ft.
       elsif( me.fueltransfert and
              !me.dependency["engineer"].getNode("cg/forward").getValue() and
              !me.dependency["engineer"].getNode("cg/aft").getValue() ) {
           if( ( me.autopilotsystem.is_engaged() and
               ( me.autopilotsystem.is_altitude_acquire() or
                 me.autopilotsystem.is_altitude_hold() ) ) or
               me.phase.is_altitude_cruise() or me.phase.is_altitude_approach() ) {
               me.fueltransfert = constant.FALSE;
               me.crewvoice.stepmember( "cgcorrect", "engineer", "flight" );
           }
       }
   } 
}

VoiceAllways.landing = func( rates ) {
   var altitudeft = me.dependency["autoflight"].getChild("altitude-select").getValue();

   if( me.selectft != altitudeft ) {
       me.selectft = altitudeft;
       me.delayselectftsec = rates;
   }
   elsif( me.delayselectftsec > 0 ) {
       if( me.delayselectftsec >= me.MODIFYSEC ) {
           if( me.crewvoice.stepmember( "goaroundset", "allways", "landing" ) ) {
               me.delayselectftsec = 0;
           }
       }
       else {
           me.delayselectftsec = me.delayselectftsec + rates;
       }
   }


   if( me.phase.is_agl_below( me.AGLLEVELFT["100ft"] ) and
       me.dependency["attitude"].getChild("indicated-pitch-deg").getValue() > me.FLAREDEG ) {
       me.crewvoice.stepmember( "attitude", "allways", "landing", constant.TRUE );
   }

   elsif( me.phase.is_agl_below( me.AGLLEVELFT["1000ft"] ) and me.phase.is_descent_final() ) {
       me.crewvoice.stepmember( "vsiexcess", "allways", "landing", constant.TRUE );
   }

   elsif( !me.automata.get_category() and me.dependency["autopilot"].getChild("land3").getValue() ) {
       me.crewvoice.stepmember( "category3", "allways", "landing" );
       me.automata.set_category( constant.TRUE );
   }

   elsif( !me.automata.get_category() and me.dependency["autopilot"].getChild("land2").getValue() ) {
       me.crewvoice.stepmember( "category2", "allways", "landing" );
       me.automata.set_category( constant.TRUE );
   }

   elsif( !me.automata.get_alert() and me.dependency["autopilot"].getChild("land2").getValue() and
          me.phase.is_agl_below_level( me.AGLLEVELFT["300ft"] ) ) {
       me.crewvoice.stepmember( "alertheight", "allways", "landing" );
       me.automata.set_alert( constant.TRUE );
   }

   elsif( !me.automata.get_decisiontogo() and
          me.phase.is_agl_below_level( me.dependency["radio-altimeter"].getChild("decision-ft").getValue() + me.AGLLEVELFT["100ft"] ) ) {
       me.crewvoice.stepmember( "100fttogo", "allways", "landing" );
       me.automata.set_decisiontogo( constant.TRUE );
   }

   elsif( me.automata.get_decisiontogo() and !me.automata.get_decision() and
          me.phase.is_agl_below_level( me.dependency["radio-altimeter"].getChild("decision-ft").getValue() ) ) {
       me.crewvoice.stepmember( "decisionheight", "allways", "landing" );
       me.automata.set_decision( constant.TRUE );
   }

   elsif( me.phase.is_agl_below( me.AGLLEVELFT["1000ft"] ) and !me.automata.get_decision() and
         ( me.phase.is_final_decrease() or
           me.phase.is_speed_below( me.sensor.Vkt( constantaero.VREFEMPTYKT,
                                                   constantaero.VREFFULLKT ) ) ) ) {
       me.crewvoice.stepmember( "approachspeed", "allways", "landing", constant.TRUE );
   }
}

VoiceAllways.check = func {
   var change = constant.FALSE;

   if( me.sensor.is_nose_change() or me.sensor.is_gear_change() ) {
       change = constant.TRUE;
       if( me.sensor.is_nose_down() and me.sensor.is_gear_down() ) {
           if( !me.crewvoice.stepmember( "5greens", "allways", "allways" ) ) {
               change = constant.FALSE;
           }
       }

       if( change ) {
           me.sensor.snapshot_nose();
       }
   }

   if( me.sensor.is_gear_change() ) {
       change = constant.TRUE;
       # on pull of lever
       if( me.sensor.is_lastgear_down() and !me.sensor.is_gear_down() ) {
           if( !me.crewvoice.stepmember( "gearup", "allways", "allways" ) ) {
               change = constant.FALSE;
           }
       }

       if( change ) {
           me.sensor.snapshot_gear();
       }
   }
}


# ==========
# VOICE CALL 
# ==========

VoiceCall = {};

VoiceCall.new = func {
   var obj = { parents : [VoiceCall],
   
               callout : nil,
               checklist : nil,
               crewvoice : nil
         };

   obj.init();
   
   return obj;
}

VoiceCall.init = func {
}

VoiceCall.set_relation = func( calloutarg, checklistarg, crewvoicearg ) {
   me.callout = calloutarg;
   me.checklist = checklistarg;
   me.crewvoice = crewvoicearg;
}

VoiceCall.pilot = func( action ) {
   var result = "";

   if( me.callout.is_holding() or me.callout.is_takeoff() or me.checklist.is_aftertakeoff() ) {
       result = me.crewvoice.stepmember( action, "pilot", "takeoff" );
   }

   elsif( me.checklist.is_beforelanding() or me.callout.is_landing() ) {
       result = me.crewvoice.stepmember( action, "pilot", "landing" );
   }

   elsif( me.callout.is_goaround() ) {
       result = me.crewvoice.stepmember( action, "pilot", "goaround" );
   }

   else {
       print("call not found : ", action);
   }

   return result;
}

VoiceCall.captain = func( action ) {
   var result = "";

   if( me.callout.is_holding() ) {
       result = me.crewvoice.stepmember( action, "captain", "takeoff" );
   }

   else {
       print("captain call not found : ", action);
   }

   return result;
}

VoiceCall.engineer = func( action ) {
   var result = "";

   if( me.callout.is_holding() or me.callout.is_takeoff() or me.checklist.is_aftertakeoff() ) {
       result = me.crewvoice.stepmember( action, "engineer", "takeoff" );
   }

   elsif( me.checklist.is_beforelanding() or me.callout.is_landing() ) {
       result = me.crewvoice.stepmember( action, "engineer", "landing" );
   }

   else {
       print("engineer call not found : ", action);
   }

   return result;
}


# ===========
# VOICE CHECK 
# ===========

VoiceCheck = {};

VoiceCheck.new = func {
   var obj = { parents : [VoiceCheck,System.new("/systems/voice")],
   
               checklist : nil,
               crewvoice : nil
         };

   obj.init();
   
   return obj;
}

VoiceCheck.init = func {
}

VoiceCheck.set_relation = func( checklistarg, crewvoicearg ) {
   me.checklist = checklistarg;
   me.crewvoice = crewvoicearg;
}

VoiceCheck.captainfeedback = func( action ) {
   me.crewvoice.nowmember( action, "captain", "allways" );
}

VoiceCheck.captain = func( action ) {
   if( me.checklist.is_beforetakeoff() ) {
       me.crewvoice.nowmember( action, "captain", "beforetakeoff" );
   }
   
   elsif( me.checklist.is_aftertakeoff() ) {
       me.crewvoice.nowmember( action, "captain", "aftertakeoff" );
   }

   elsif( me.checklist.is_afterstart() ) {
       me.crewvoice.nowmember( action, "captain", "afterstart" );
   }

   elsif( me.checklist.is_taxi() ) {
       me.crewvoice.nowmember( action, "captain", "taxi" );
   }

   elsif( me.checklist.is_afterlanding() ) {
       me.crewvoice.nowmember( action, "captain", "afterlanding" );
   }

   else {
       print("captain check not found : ", action);
   }
}

VoiceCheck.pilot = func( action, argument ) {
   me.itself["root"].getChild("argument").setValue( argument );

   if( me.checklist.is_beforestart() ) {
       me.crewvoice.nowmember( action, "pilot", "beforestart" );
   }

   elsif( me.checklist.is_pushback() ) {
       me.crewvoice.nowmember( action, "pilot", "pushback" );
   }

   elsif( me.checklist.is_afterstart() ) {
       me.crewvoice.nowmember( action, "pilot", "afterstart" );
   }

   else {
       print("pilot check not found : ", action);
   }
}

VoiceCheck.engineer = func( action ) {
   if( me.checklist.is_aftertakeoff() ) {
       me.crewvoice.nowmember( action, "engineer", "aftertakeoff" );
   }

   elsif( me.checklist.is_climb() ) {
       me.crewvoice.nowmember( action, "engineer", "climb" );
   }

   elsif( me.checklist.is_transsonic() ) {
       me.crewvoice.nowmember( action, "engineer", "transsonic" );
   }

   elsif( me.checklist.is_descent() ) {
       me.crewvoice.nowmember( action, "engineer", "descent" );
   }

   elsif( me.checklist.is_approach() ) {
       me.crewvoice.nowmember( action, "engineer", "approach" );
   }

   elsif( me.checklist.is_beforelanding() ) {
       me.crewvoice.nowmember( action, "engineer", "beforelanding" );
   }

   elsif( me.checklist.is_afterlanding() ) {
       me.crewvoice.nowmember( action, "engineer", "afterlanding" );
   }

   elsif( me.checklist.is_parking() ) {
       me.crewvoice.nowmember( action, "engineer", "parking" );
   }

   elsif( me.checklist.is_stopover() ) {
       me.crewvoice.nowmember( action, "engineer", "stopover" );
   }

   elsif( me.checklist.is_cockpit() ) {
       me.crewvoice.nowmember( action, "engineer", "cockpit" );
   }

   elsif( me.checklist.is_beforestart() ) {
       me.crewvoice.nowmember( action, "engineer", "beforestart" );
   }

   elsif( me.checklist.is_pushback() ) {
       me.crewvoice.nowmember( action, "engineer", "pushback" );
   }

   elsif( me.checklist.is_taxi() ) {
       me.crewvoice.nowmember( action, "engineer", "taxi" );
   }

   elsif( me.checklist.is_beforetakeoff() ) {
       me.crewvoice.nowmember( action, "engineer", "beforetakeoff" );
   }

   else {
       print("engineer check not found : ", action);
   }
}


# ===============
# VOICE CHECKLIST 
# ===============

VoiceChecklist = {};

VoiceChecklist.new = func {
   var obj = { parents : [VoiceChecklist,System.new("/systems/voice")],
   
               checklist : nil,
               
               lastcheck : Checklist.new("/systems/voice"),

               real : constant.FALSE                             # real checklist
         };

   obj.init();
   
   return obj;
}

VoiceChecklist.init = func {
}

VoiceChecklist.set_relation = func( checklistarg ) {
   me.checklist = checklistarg;
}

VoiceChecklist.sendchecklist = func {
   me.itself["root"].getChild("checklist").setValue(me.checklist.get_checklist());
   me.itself["root"].getChild("real").setValue(me.real);
}

VoiceChecklist.sendemergency = func {
   me.itself["root"].getChild("emergency").setValue(me.checklist.get_emergency());
   me.itself["root"].getChild("real").setValue(me.real);
}

VoiceChecklist.checklistinit = func( state, real ) {
   me.checklist.set_checklist( state );
   me.real = real;

   # red : processing
   if( me.real ) {
       me.itself["display"].getChild("processing").setValue(me.checklist.get_checklist());
   }

   else {
       var processing = me.itself["display"].getChild("processing").getValue();

       # blue : completed
       if( processing != "" ) {
           me.lastcheck.set_checklist( processing );

           me.itself["display"].getChild("processing").setValue("");
           me.itself["display"].getChild("completed").setValue(me.lastcheck.get_checklist());
       }
   }

   me.checklist.unset_completed();

   me.sendchecklist();
}

VoiceChecklist.emergencyinit = func( state, real ) {
   me.checklist.set_emergency( state );
   me.real = real;

   # red : processing
   if( me.real ) {
       me.itself["display"].getChild("processing").setValue(me.checklist.get_emergency());
   }

   else {
       var processing = me.itself["display"].getChild("processing").getValue();

       # blue : completed
       if( processing != "" ) {
           me.itself["display"].getChild("processing").setValue("");
           me.itself["display"].getChild("completed").setValue(processing);
       }
   }

   me.checklist.unset_completed();

   me.sendemergency();
}

VoiceChecklist.is_climb = func {
   return me.lastcheck.is_climb();
}

VoiceChecklist.is_approach = func {
   return me.lastcheck.is_approach();
}

VoiceChecklist.is_beforelanding = func {
   return me.lastcheck.is_beforelanding();
}

VoiceChecklist.noemergencyinit = func() {
   me.emergencyinit( "", constant.FALSE, me.checklist );
}

VoiceChecklist.nochecklistinit = func() {
   me.checklistinit( "", constant.FALSE );
}

VoiceChecklist.afterlandinginit = func {
   var result = constant.TRUE;

   if( me.checklist.is_afterlanding() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.checklistinit( "afterlanding", constant.TRUE );
   }
}

VoiceChecklist.parkinginit = func {
   var result = constant.TRUE;

   if( me.checklist.is_parking() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.checklistinit( "parking", constant.TRUE );
   }
}

VoiceChecklist.stopoverinit = func {
   var result = constant.TRUE;

   if( me.checklist.is_stopover() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.checklistinit( "stopover", constant.TRUE );
   }
}

VoiceChecklist.externalinit = func {
   var result = constant.TRUE;

   if( me.checklist.is_external() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.checklistinit( "external", constant.TRUE );
   }
}

VoiceChecklist.preliminaryinit = func {
   var result = constant.TRUE;

   if( me.checklist.is_preliminary() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.checklistinit( "preliminary", constant.TRUE );
   }
}

VoiceChecklist.cockpitinit = func {
   var result = constant.TRUE;

   if( me.checklist.is_cockpit() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.checklistinit( "cockpit", constant.TRUE );
   }
}

VoiceChecklist.beforestartinit = func {
   var result = constant.TRUE;

   if( me.checklist.is_beforestart() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.checklistinit( "beforestart", constant.TRUE );
   }
}

VoiceChecklist.enginestartinit = func {
   var result = constant.TRUE;

   if( me.checklist.is_enginestart() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.checklistinit( "enginestart", constant.TRUE );
   }
}

VoiceChecklist.pushbackinit = func {
   var result = constant.TRUE;

   if( me.checklist.is_pushback() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.checklistinit( "pushback", constant.TRUE );
   }
}

VoiceChecklist.startedinit = func {
   var result = constant.TRUE;

   if( me.checklist.is_started() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.checklistinit( "started", constant.FALSE );
   }
}

VoiceChecklist.afterstartinit = func {
   var result = constant.TRUE;

   if( me.checklist.is_afterstart() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.checklistinit( "afterstart", constant.TRUE );
   }
}

VoiceChecklist.taxiinit = func {
   var result = constant.TRUE;

   if( me.checklist.is_taxi() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.checklistinit( "taxi", constant.TRUE );
   }
}

VoiceChecklist.runwayinit = func {
   var result = constant.TRUE;

   if( me.checklist.is_runway() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.checklistinit( "runway", constant.FALSE );
   }
}

VoiceChecklist.beforetakeoffinit = func {
   var result = constant.TRUE;

   if( me.checklist.is_beforetakeoff() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.checklistinit( "beforetakeoff", constant.TRUE );
   }
}

VoiceChecklist.aftertakeoffinit = func {
   var result = constant.TRUE;

   if( me.checklist.is_aftertakeoff() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.checklistinit( "aftertakeoff", constant.TRUE );
   }
}

VoiceChecklist.climbinit = func {
   var result = constant.TRUE;

   if( me.checklist.is_climb() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.checklistinit( "climb", constant.TRUE );
   }
}

VoiceChecklist.transsonicinit = func {
   var result = constant.TRUE;

   if( me.checklist.is_transsonic() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.checklistinit( "transsonic", constant.TRUE );
   }
}

VoiceChecklist.cruiseclimbinit = func {
   var result = constant.TRUE;

   if( me.checklist.is_cruiseclimb() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.checklistinit( "cruiseclimb", constant.FALSE );
   }
}

VoiceChecklist.descentinit = func {
   var result = constant.TRUE;

   if( me.checklist.is_descent() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.checklistinit( "descent", constant.TRUE );
   }
}

VoiceChecklist.approachinit = func {
   var result = constant.TRUE;

   if( me.checklist.is_approach() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.checklistinit( "approach", constant.TRUE );
   }
}

VoiceChecklist.beforelandinginit = func {
   var result = constant.TRUE;

   if( me.checklist.is_beforelanding() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.checklistinit( "beforelanding", constant.TRUE );
   }
}

VoiceChecklist.fourengineflameoutinit = func {
   var result = constant.TRUE;

   if( me.checklist.is_fourengineflameout() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.emergencyinit( "fourengineflameout", constant.TRUE );
   }
}

VoiceChecklist.fourengineflameoutmach1init = func {
   var result = constant.TRUE;

   if( me.checklist.is_fourengineflameoutmach1() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.emergencyinit( "fourengineflameoutmach1", constant.TRUE );
   }
}


# ==============
# VOICE AUTOMATA 
# ==============

VoiceAutomata = {};

VoiceAutomata.new = func {
   var obj = { parents : [VoiceAutomata,System.new("/systems/voice")],

               phase : nil,
               callout : nil,
               
               category : constant.FALSE,
               alert : constant.FALSE,
               decision : constant.FALSE,
               decisiontogo : constant.FALSE,

               v1 : constant.FALSE,
               v2 : constant.FALSE,

               automata : "",
               automata2 : ""
         };

   obj.init();
   
   return obj;
}

VoiceAutomata.init = func {
}

VoiceAutomata.set_relation = func( phasearg, calloutarg ) {
   me.phase = phasearg;
   me.callout = calloutarg;
}

VoiceAutomata.set_v1 = func( reached ) {
   me.v1 = reached;
}

VoiceAutomata.set_v2 = func( reached ) {
   me.v2 = reached;
}

VoiceAutomata.get_v1 = func {
   return me.v1;
}

VoiceAutomata.get_v2 = func {
   return me.v2;
}

VoiceAutomata.set_category = func( reached ) {
   me.category = reached;
}

VoiceAutomata.set_alert = func( reached ) {
   me.alert = reached;
}

VoiceAutomata.set_decision = func( reached ) {
   me.decision = reached;
}

VoiceAutomata.set_decisiontogo = func( reached ) {
   me.decisiontogo = reached;
}

VoiceAutomata.get_category = func {
   return me.category;
}

VoiceAutomata.get_alert = func {
   return me.alert;
}

VoiceAutomata.get_decision = func {
   return me.decision;
}

VoiceAutomata.get_decisiontogo = func {
   return me.decisiontogo;
}

VoiceAutomata.sendcallout = func( state2, state3 ) {
   me.automata = state2;
   me.automata2 = state3;
   
   me.itself["automata"][0].setValue( me.automata );
   me.itself["automata"][1].setValue( me.automata2 );
}

VoiceAutomata.calloutinit = func( state, state2, state3 ) {
   me.phase.set_level();

   me.callout.set_callout( state );
   me.callout.send();
   me.sendcallout( state2, state3 );
}

VoiceAutomata.isStep = func( step ) {
   var result = constant.FALSE;
   
   if( me.automata == step ) {
       result = constant.TRUE;
   }
   
   return result;
}

VoiceAutomata.isStep2 = func( step ) {
   var result = constant.FALSE;
   
   if( me.automata2 == step ) {
       result = constant.TRUE;
   }
   
   return result;
}

VoiceAutomata.setStep = func( step ) {
   me.automata = step;
}

VoiceAutomata.setStep2 = func( step ) {
   me.automata2 = step;
}

VoiceAutomata.nocalloutinit = func {
   me.calloutinit( "voice is disabled", "", "", me.phase, me.callout );
}

VoiceAutomata.taxiwayinit = func {
   var result = constant.TRUE;

   if( me.callout.is_taxiway() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.calloutinit( "taxiway", "", "" );
   }
}

VoiceAutomata.terminalinit = func {
   var result = constant.TRUE;

   if( me.callout.is_terminal() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.calloutinit( "terminal", "", "" );
   }
}

VoiceAutomata.gateinit = func {
   var result = constant.TRUE;

   if( me.callout.is_gate() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.calloutinit( "gate", "", "" );
   }
}

VoiceAutomata.holdinginit = func {
   var result = constant.TRUE;

   if( me.callout.is_holding() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.calloutinit( "holding", "holding", "holding" );
   }
}

VoiceAutomata.takeoffinit = func ( overwrite = 0 ) {
   var result = constant.TRUE;

   if( me.callout.is_takeoff() ) {
       result = constant.FALSE;
   }

   if( result or overwrite ) {
       me.calloutinit( "takeoff", "takeoff", "takeoff" );

       me.v1 = constant.FALSE;
       me.v2 = constant.FALSE;
   }
}

VoiceAutomata.flightinit = func {
   var result = constant.TRUE;

   if( me.callout.is_flight() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.calloutinit( "flight", "", "" );
   }
}

VoiceAutomata.landinginit = func {
   var result = constant.TRUE;

   if( me.callout.is_landing() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.calloutinit( "landing", "landing", "landing" );

       me.category = constant.FALSE;
       me.alert = constant.FALSE;
       me.decision = constant.FALSE;
       me.decisiontogo = constant.FALSE;
   }
}

VoiceAutomata.goaroundinit = func {
   var result = constant.TRUE;

   if( me.callout.is_goaround() ) {
       result = constant.FALSE;
   }

   if( result ) {
       me.calloutinit( "goaround", "goaround", "goaround" );
   }
}


# ========
# VOICE AI 
# ========

VoiceAI = {};

VoiceAI.new = func {
   var obj = { parents : [VoiceAI,System.new("/systems/voice")]
         };

   return obj;
}

VoiceAI.has_AI = func {
   var result = constant.FALSE;

   # AI triggers flight checklists
   if( me.dependency["crew-ctrl"].getChild("checklist").getValue() ) {
       result = me.has_crew();
   }

   return result;
}

VoiceAI.has_crew = func {
   var result = constant.FALSE;

   if( me.itself["root"].getChild("serviceable").getValue() ) {
       if( me.dependency["copilot-ctrl"].getChild("activ").getValue() and
           me.dependency["engineer-ctrl"].getChild("activ").getValue() ) {
           result = constant.TRUE;
       }
   }

   return result;
}


# ==========
# VOICE WORD 
# ==========

Voiceword = {};

Voiceword.new = func {
   var obj = { parents : [Voiceword,System.new("/systems/voice")],

               # pilot in command
               captain :  { "takeoff" : {}, "afterstart" : {}, "taxi" : {}, "beforetakeoff" : {}, "aftertakeoff" : {}, "afterlanding" : {}, "allways" : {} },

               # pilot not in command
               pilot :  { "takeoff" : {}, "climb" : {}, "landing" : {}, "goaround" : {}, "beforestart" : {}, "pushback" : {}, "afterstart" : {} },
               
               allways : { "takeoff" : {}, "landing" : {}, "flight" : {}, "allways" : {} },

               engineer : { "takeoff" : {}, "flight" : {}, "climb" : {}, "transsonic" : {}, "descent" : {}, "approach" : {}, "landing" : {},
                            "beforelanding" : {}, "afterlanding" : {}, "parking" : {}, "stopover" : {}, "cockpit" : {}, "beforestart" : {},
                            "pushback" : {}, "taxi" : {}, "beforetakeoff" : {}, "aftertakeoff" : {} }
         };

   obj.init();
   
   return obj;
}

Voiceword.init = func {
   me.inittext();
}

Voiceword.inittable = func( path, table ) {
   var key = "";
   var text = "";
   var node = props.globals.getNode(path).getChildren("message");

   for( var i=0; i < size(node); i=i+1 ) {
        key = node[i].getChild("action").getValue();
        text = node[i].getChild("text").getValue();
        table[key] = text;
   }
}

Voiceword.inittext = func {
   me.inittable(me.itself["checklist"].getNode("beforetakeoff/engineer[0]").getPath(), me.engineer["beforetakeoff"] );
   me.inittable(me.itself["checklist"].getNode("beforetakeoff/captain[0]").getPath(), me.captain["beforetakeoff"] );

   me.inittable(me.itself["callout"].getNode("takeoff/captain").getPath(), me.captain["takeoff"] );

   me.inittable(me.itself["callout"].getNode("takeoff/pilot[0]").getPath(), me.pilot["takeoff"] );
   me.inittable(me.itself["callout"].getNode("takeoff/pilot[1]").getPath(), me.pilot["climb"] );
   me.inittable(me.itself["callout"].getNode("takeoff/pilot[2]").getPath(), me.allways["takeoff"] );
   me.inittable(me.itself["callout"].getNode("takeoff/engineer[0]").getPath(), me.engineer["takeoff"] );

   me.inittable(me.itself["checklist"].getNode("aftertakeoff/engineer[0]").getPath(), me.engineer["aftertakeoff"] );
   me.inittable(me.itself["checklist"].getNode("aftertakeoff/captain[0]").getPath(), me.captain["aftertakeoff"] );

   me.inittable(me.itself["callout"].getNode("flight/pilot[0]").getPath(), me.allways["flight"] );
   me.inittable(me.itself["callout"].getNode("flight/engineer[0]").getPath(), me.engineer["flight"] );

   me.inittable(me.itself["checklist"].getNode("climb/engineer[0]").getPath(), me.engineer["climb"] );

   me.inittable(me.itself["checklist"].getNode("transsonic/engineer[0]").getPath(), me.engineer["transsonic"] );

   me.inittable(me.itself["checklist"].getNode("descent/engineer[0]").getPath(), me.engineer["descent"] );

   me.inittable(me.itself["checklist"].getNode("approach/engineer[0]").getPath(), me.engineer["approach"] );

   me.inittable(me.itself["checklist"].getNode("beforelanding/engineer").getPath(), me.engineer["beforelanding"] );

   me.inittable(me.itself["callout"].getNode("landing/pilot[0]").getPath(), me.pilot["landing"] );
   me.inittable(me.itself["callout"].getNode("landing/pilot[1]").getPath(), me.allways["landing"] );
   me.inittable(me.itself["callout"].getNode("landing/engineer[0]").getPath(), me.engineer["landing"] );

   me.inittable(me.itself["callout"].getNode("goaround/pilot[0]").getPath(), me.pilot["goaround"] );

   me.inittable(me.itself["checklist"].getNode("afterlanding/engineer[0]").getPath(), me.engineer["afterlanding"] );
   me.inittable(me.itself["checklist"].getNode("afterlanding/captain[0]").getPath(), me.captain["afterlanding"] );

   me.inittable(me.itself["checklist"].getNode("parking/engineer[0]").getPath(), me.engineer["parking"] );

   me.inittable(me.itself["checklist"].getNode("stopover/engineer[0]").getPath(), me.engineer["stopover"] );

   me.inittable(me.itself["checklist"].getNode("cockpit/engineer[0]").getPath(), me.engineer["cockpit"] );

   me.inittable(me.itself["checklist"].getNode("beforestart/pilot[0]").getPath(), me.pilot["beforestart"] );
   me.inittable(me.itself["checklist"].getNode("beforestart/engineer[0]").getPath(), me.engineer["beforestart"] );

   me.inittable(me.itself["checklist"].getNode("pushback/pilot[0]").getPath(), me.pilot["pushback"] );
   me.inittable(me.itself["checklist"].getNode("pushback/engineer[0]").getPath(), me.engineer["pushback"] );

   me.inittable(me.itself["checklist"].getNode("afterstart/pilot[0]").getPath(), me.pilot["afterstart"] );
   me.inittable(me.itself["checklist"].getNode("afterstart/captain[0]").getPath(), me.captain["afterstart"] );

   me.inittable(me.itself["checklist"].getNode("taxi/engineer[0]").getPath(), me.engineer["taxi"] );
   me.inittable(me.itself["checklist"].getNode("taxi/captain[0]").getPath(), me.captain["taxi"] );

   me.inittable(me.itself["checklist"].getNode("all/captain[0]").getPath(), me.captain["allways"] );

   me.inittable(me.itself["callout"].getNode("all/pilot[0]").getPath(), me.allways["allways"] );
}

Voiceword.get_word = func( member, phase ) {
   if( member == "captain" ) {
       return me.captain[phase];
   }
   
   elsif( member == "pilot" ) {
       return me.pilot[phase];
   }
   
   elsif( member == "allways" ) {
       return me.allways[phase];
   }
   
   elsif( member == "engineer" ) {
       return me.engineer[phase];
   }
   
   else {
       print( "word " ~ phase ~ " not found for " ~ member );
       return "";
   }
}


# ==========
# CREW VOICE 
# ==========

Crewvoice = {};

Crewvoice.new = func {
   var obj = { parents : [Crewvoice,System.new("/systems/voice")],

               word : Voiceword.new(),
               voicebox : Voicebox.new(),
               
               CONVERSATIONSEC : 4.0,                            # until next message
               REPEATSEC : 4.0,                                  # between 2 messages

               # pilot in command
               phrasecaptain : "",
               delaycaptainsec : 0.0,

               # pilot not in command
               phrasepilot : "",
               delaypilotsec : 0.0,                              # delay this phrase
               nextsec : 0.0,                                    # delay the next phrase

               # engineer
               phraseengineer : "",
               delayengineersec : 0.0,

               asynchronous : constant.FALSE,

               hearvoice : constant.FALSE
         };

   obj.init();

   return obj;
}

Crewvoice.init = func {
   # translate phrase into sound
   me.hearvoice = me.itself["sound"].getChild("enabled").getValue();
}

Crewvoice.textexport = func {
   var feedback = me.voicebox.textexport();

   # also to test sound
   if( me.voicebox.is_on() ) {
       me.talkpilot( feedback );
   }
   else {
       me.talkengineer( feedback );
   }
}

Crewvoice.schedule = func {
   me.voicebox.schedule();
}

Crewvoice.stepmember = func( state, member, phase, repeat = 0 ) {
   if( member == "pilot" ) {
       me.steppilot( state, me.word.get_word( member, phase ) );
   }
   
   elsif( member == "engineer" ) {
       me.stepengineer( state, me.word.get_word( member, phase ) );
    }
   
   elsif( member == "captain" ) {
       me.stepcaptain( state, me.word.get_word( member, phase ) );
   }
   
   elsif( member == "allways" ) {
       me.stepallways( state, me.word.get_word( member, phase ), repeat );
   }
   
   else {
       print( "step " ~ state ~ " not found for " ~ member ~ " at " ~ phase );
   }
}

Crewvoice.nowmember = func( state, member, phase ) {
   var found = constant.TRUE;
   
   if( member == "pilot" ) {
       me.steppilot( state, me.word.get_word( member, phase ) );
   }
   
   elsif( member == "engineer" ) {
       me.stepengineer( state, me.word.get_word( member, phase ) );
    }
   
   elsif( member == "captain" ) {
       me.stepcaptain( state, me.word.get_word( member, phase ) );
   }
   
   else {
       found = constant.FALSE;
       print( "now " ~ state ~ " not found for " ~ member ~ " at " ~ phase );
   }

   if( found ) {
       me.playvoices( constant.HUMANSEC );
   }
}

Crewvoice.stepallways = func( state, table, repeat = 0 ) {
   var result = constant.FALSE;

   if( !me.asynchronous ) {
       if( me.nextsec <= 0 ) {
           me.phrasepilot = table[state];
           me.delaypilotsec = 0;

           if( repeat ) {
               me.nextsec = me.REPEATSEC;
           }

           if( me.phrasepilot == "" ) {
               print("missing voice text : ",state);
           }

           me.asynchronous = constant.TRUE;
           result = constant.TRUE;
       }
   }

   return result;
}

Crewvoice.steppilot = func( state, table ) {
   me.talkpilot( table[state] );

   if( me.phrasepilot == "" ) {
       print("missing voice text : ",state);
   }

   me.asynchronous = constant.TRUE;

   return state;
}

Crewvoice.talkpilot = func( phrase ) {
   if( me.phrasepilot != "" ) {
       print("phrase overflow : ", phrase);
   }

   # add an optional argument
   if( find("%s", phrase) >= 0 ) {
       phrase = sprintf( phrase, me.itself["root"].getChild("argument").getValue() );
   }

   me.phrasepilot = phrase;
   me.delaypilotsec = 0;
}

Crewvoice.stepengineer = func( state, table ) {
   me.talkengineer( table[state] );

   if( me.phraseengineer == "" ) {
       print("missing voice text : ",state);
   }

   return state;
}

Crewvoice.talkengineer = func( phrase ) {
   if( me.phraseengineer != "" ) {
       print("engineer phrase overflow : ", phrase);
   }

   me.phraseengineer = phrase;
   me.delayengineersec = 0;
}

Crewvoice.stepcaptain = func( state, table ) {
   me.talkcaptain( table[state] );

   if( me.phrasecaptain == "" ) {
       print("missing voice text : ",state);
   }

   me.asynchronous = constant.TRUE;

   return state;
}

Crewvoice.talkcaptain = func( phrase ) {
   if( me.phrasecaptain != "" ) {
       print("captain phrase overflow : ", phrase);
   }

   me.phrasecaptain = phrase;
   me.delaycaptainsec = 0;
}

Crewvoice.willplay = func {
   var result = constant.FALSE;

   if( me.phrasepilot != "" or me.phraseengineer != "" or me.phrasecaptain != "" ) {
       result = constant.TRUE;
   }

   return result;
}

Crewvoice.is_asynchronous = func {
   return me.asynchronous;
}

Crewvoice.playvoices = func( rates ) {
   # pilot not in command calls out
   if( me.delaypilotsec <= 0 ) {
       if( me.phrasepilot != "" ) {
           me.itself["display"].getChild("copilot").setValue(me.phrasepilot);

           me.sendvoices( me.phrasepilot, "copilot" );
           me.voicebox.sendtext(me.phrasepilot);
           me.phrasepilot = "";

           # engineer lets pilot speak
           if( me.phraseengineer != "" ) {
               me.delayengineersec = me.CONVERSATIONSEC;
           }
        }
   }
   else {
       me.delaypilotsec = me.delaypilotsec - rates;
   }

   # no engineer voice yet
   if( me.delayengineersec <= 0 ) {
       if( me.phraseengineer != "" ) {
           me.itself["display"].getChild("engineer").setValue(me.phraseengineer);

           me.sendvoices( me.phraseengineer, "engineer" );
           me.voicebox.sendtext(me.phraseengineer, constant.TRUE);
           me.phraseengineer = "";
       }
   }
   else {
       me.delayengineersec = me.delayengineersec - rates;
   }

   # pilot in command calls out
   if( me.delaycaptainsec <= 0 ) {
       if( me.phrasecaptain != "" ) {
           me.itself["display"].getChild("captain").setValue(me.phrasecaptain);

           me.sendvoices( me.phrasecaptain, "captain" );
           me.voicebox.sendtext(me.phrasecaptain, constant.FALSE, constant.TRUE);
           me.phrasecaptain = "";
       }
   }
   else {
       me.delaycaptainsec = me.delaycaptainsec - rates;
   }

   if( me.nextsec > 0 ) {
       me.nextsec = me.nextsec - rates;
   }

   me.asynchronous = constant.FALSE;
}

Crewvoice.sendvoices = func( sentphrase, sentcrew ) {
   if( me.hearvoice ) {
       var voice = "pilot";
       
       if( sentcrew == "copilot" ) {
           voice = "copilot";
       }
       
       me.itself["sound"].getChild(voice).setValue(sentphrase);
   }
}
