/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include "buffer_tile.h"

#define DEFAULT_TILE_WIDTH 64
#define DEFAULT_TILE_HEIGHT 64


#define DATA   (&tile_buffer->tiles[i].data)
#define VALID  (tile_buffer->tiles[i].valid)
#define VALID2 (tile_buffer->tiles[i].validating)

#define tilenum(b,x,y) ((b)->_x.tile.count * (y) + (x))


static void
gimp_tile_buffer_delete (GimpBuffer * buffer)
{
  GimpTileBuffer * tile_buffer = GIMP_TILE_BUFFER (buffer);
  gint i, j;

  j = buffer->_x.tile.count * buffer->_y.tile.count;
  for (i = 0; i < j; i++)
    {
      d_uninit (DATA);
    }
  g_free (tile_buffer->tiles);
  g_free (tile_buffer);
}


static gboolean
gimp_tile_buffer_alloc (GimpBuffer   *buffer,
                        GimpArea     *area,
                        Alloc         how)
{
  GimpTileBuffer * tile_buffer = GIMP_TILE_BUFFER (buffer);
  gint fails = 0;
  GimpPoint p;
  gint n;

  /* common tiles everyone initially shares */
  static GimpMemPtr junk[5];
  static int need_init = 1;

  /* set up the gray50 tiles */
  if (need_init)
    {
      gint i;
      need_init = 0;
      for (i = 1; i < 5; i++)
        {
          n = (DEFAULT_TILE_WIDTH * DEFAULT_TILE_HEIGHT * i);
          d_init (&junk[i]);
          d_alloc (&junk[i], n);
          d_write (&junk[i]);
          memset (guchar_d (&junk[i]), 128, n);
          d_release (&junk[i]);
        }      
    }

  n = gimp_buffer_bpp (buffer);

  for (p.y = area->a.y; p.y < area->b.y; p.y++)
    {
      for (p.x = area->a.x; p.x < area->b.x; p.x++)
        {
          gint i = tilenum (buffer, p.x, p.y);
          
          switch (how)
            {
            case ALLOC_ALLOC:
              if (! d_is_alloced (DATA))
                {
                  if (d_join (&junk[n], DATA) != TRUE)
                    {
                      fails++;
                    }
                }
              break;
              
            case ALLOC_UNALLOC:
              if (d_is_alloced (DATA))
                {
                  if (d_unalloc (DATA) != TRUE)
                    {
                      fails++;
                    }
                  VALID = FALSE;
                }
              break;
              
            case ALLOC_NONE:
              g_warning ("bad value");
              return FALSE;
            }
        }
    }
  
  if (fails != 0)
    g_warning ("%d tiles failed alloc", fails);

  return TRUE;
}


static gboolean
gimp_tile_buffer_map (GimpBuffer    *dst,
                      GimpArea      *darea,
                      GimpBuffer    *src,
                      GimpArea      *sarea)
{
  GimpTileBuffer * db = GIMP_TILE_BUFFER (dst);
  GimpTileBuffer * sb = GIMP_TILE_BUFFER (src);
  GimpPoint sp, dp;
  GimpTile *s, *d;
  gint fails = 0;
  

  for (dp.y = darea->a.y, sp.y = sarea->a.y;
       dp.y < darea->b.y;
       dp.y++, sp.y++)
    {
      if (sp.y == sarea->b.y)
        sp.y = sarea->a.y;

      for (dp.x = darea->a.x, sp.x = sarea->a.x;
           dp.x < darea->b.x;
           dp.x++, sp.x++)
        {
          if (sp.x == sarea->b.x)
            sp.x = sarea->a.x;

          s = &sb->tiles[tilenum (src, sp.x, sp.y)];
          if (s->validating == TRUE)
            {
              fails++;
              continue;
            }

          d = &db->tiles[tilenum (dst, dp.x, dp.y)];
          if (d->validating == TRUE)
            {
              fails++;
              continue;
            }

          /* happens when we map an area from an internal tile */
          if (&s->data == &d->data)
            {
              continue;
            }
          
          d_uninit (&d->data);
          if (d_join (&s->data, &d->data) != TRUE)
            {
              fails++;
              continue;
            }

          d->valid = s->valid;
        }
    }

  if (fails != 0)
    g_warning ("%d tiles failed map", fails);
  
  return TRUE;
}


static gboolean
gimp_tile_buffer_validate (GimpBuffer  *buffer,
                           GimpArea    *area, 
                           Validate     how)
{
  GimpTileBuffer * tile_buffer = GIMP_TILE_BUFFER (buffer);
  gint fails = 0;
  GimpPoint p;
  GimpArea a;
  

   for (p.y = area->a.y; p.y < area->b.y; p.y++)
    {
      for (p.x = area->a.x; p.x < area->b.x; p.x++)
        {
          gint i = tilenum (buffer, p.x, p.y);

          switch (how)
            {
            case VALIDATE_VALIDATE:
              if (VALID == TRUE)
                break;

              /* happens if vfunc refs the tile */
              if (VALID2 == TRUE)
                {
                  g_warning ("recursive validate");
                  fails++;
                  break;
                }

              a.a = p;
              a.b = p;
              a.b.x++;
              a.b.y++;
                  
              if (buffer->vfunc == NULL)
                {
                  gimp_tile_buffer_alloc (buffer, &a, ALLOC_UNALLOC);
                  gimp_tile_buffer_alloc (buffer, &a, ALLOC_ALLOC);
                  VALID = TRUE;
                }
              else
                {
                  if (! d_is_alloced (DATA))
                    {
                      gimp_tile_buffer_alloc (buffer, &a, ALLOC_ALLOC);
                    }
                  
                  if (d_write (DATA) != TRUE)
                    {
                      fails++;
                      break;
                    }

                  VALID2 = TRUE;
                  {
                    GimpArea a2 = a;

                    a2.a.x *= buffer->_x.tile.size;
                    a2.a.y *= buffer->_y.tile.size;
                    a2.b.x *= buffer->_x.tile.size;
                    a2.b.y *= buffer->_y.tile.size;

                    buffer->vfunc (buffer, &a2, guchar_d (DATA));
                  }
                  VALID2 = FALSE;

                  d_release (DATA);
                  VALID = TRUE;
                }
              break;
              

            case VALIDATE_INVALIDATE:
              if (d_usecount (DATA) == 0)
                {
                  VALID = FALSE;
                }
              else
                {
                  g_warning ("in use");
                  fails++;
                }
              break;

            case VALIDATE_NONE:
              g_warning ("bad value");
              return FALSE;
            }
        }
    }
  
  if (fails != 0)
    g_warning ("%d tiles failed valid", fails);

  return TRUE;
}



static GimpMemPtr *
gimp_tile_buffer_use (GimpBuffer  *buffer,
                      GimpPoint   *tile, 
                      Use          how)
{
  GimpTileBuffer * tile_buffer = GIMP_TILE_BUFFER (buffer);
  gint i = tilenum (buffer, tile->x, tile->y);

  if ((VALID != TRUE) && ((how == USE_READ) ||
                          (how == USE_UPDATE) ||
                          (how == USE_WRITE)))
    {
      GimpArea area;

      area.a = *tile;
      area.b = *tile;
      area.b.x++;
      area.b.y++;
      
      if (gimp_tile_buffer_validate (buffer, &area,
                                     VALIDATE_VALIDATE) != TRUE)
        {
          g_warning ("tilebuffer autovalidate failed");
          return NULL;
        }
    }

  switch (how)
    {
    case USE_READ:
      g_return_val_if_fail (d_read (DATA) == TRUE, NULL);
      break;

    case USE_UPDATE:
      g_return_val_if_fail (d_update (DATA) == TRUE, NULL);
      break;

    case USE_WRITE:
      g_return_val_if_fail (d_write (DATA) == TRUE, NULL);
      break;

    case USE_RELEASE:
      g_return_val_if_fail (d_release (DATA) == TRUE, NULL);
      break;

    case USE_NONE:
      break;
    }

  return DATA;
}


static gboolean
gimp_tile_buffer_valid (GimpBuffer    *buffer,
                        GimpPoint     *tile)
{
  GimpTileBuffer * tile_buffer = GIMP_TILE_BUFFER (buffer);
  gint i = tilenum (buffer, tile->x, tile->y);

  return VALID;
}






GimpTileBuffer *
gimp_tile_buffer_new (Tag   tag,
                      gint  width,
                      gint  height,
                      gint  tile_width,
                      gint  tile_height)
{
  GimpTileBuffer * tile_buffer;
  GimpBuffer * buffer;
  gint i, j;

  static GimpTileBufferClass my_class =
  {
    {
      BUFFER_TILED,
      gimp_tile_buffer_delete,
      gimp_tile_buffer_alloc,
      gimp_tile_buffer_map,
      gimp_tile_buffer_validate,
      gimp_tile_buffer_use,
      gimp_tile_buffer_valid
    }
  };

  if (tile_width == 0)
    tile_width = DEFAULT_TILE_WIDTH;

  if (tile_height == 0)
    tile_height = DEFAULT_TILE_HEIGHT;

  tile_buffer = g_new (GimpTileBuffer, 1);
  buffer = GIMP_BUFFER (tile_buffer);

  buffer->klass = (void*) &my_class;

  buffer->vfunc = NULL;
  buffer->vdata = NULL;
  
  buffer->_x.tile.count = (width + tile_width - 1) / tile_width;
  buffer->_x.tile.size = tile_width;
  buffer->_x.image.size = width;
  buffer->_x.image.offset = 0;

  buffer->_y.tile.count = (height + tile_height - 1) / tile_height;
  buffer->_y.tile.size = tile_height;
  buffer->_y.image.size = height;
  buffer->_y.image.offset = 0;

  buffer->_z.tag = tag;
  buffer->_z.bpp = tag_bytes (tag);

  j = buffer->_x.tile.count * buffer->_y.tile.count;

  tile_buffer->tiles = g_new (GimpTile, j);
  
  for (i = 0; i < j; i++)
    {
      d_init (DATA);
      VALID = FALSE;
      VALID2 = FALSE;
    }

  buffer->qq_x = 0;
  buffer->qq_y = 0;

  return tile_buffer;
}


