/*  GUBI - Gtk+ User Interface Builder
 *  Copyright (C) 1997  Tim Janik
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include	"RCS.h"
RCS_ID("$Id: structures.c,v 1.7 1997/05/20 21:17:50 tim Exp $")


#define		__structures_c__

#include	"structures.h"
#include	"widdata.h"
#include	"misc.h"
#include	"defines.h"
#include	<string.h>
#include	<ctype.h>


/* --- widget info tree --- */
#include	"widdata.dat"



/* --- defines --- */
#define	FIELD_VALUE_U(struct_begin,offset)	\
	((field_value_U*)((guchar*)struct_begin+offset))



/* --- functions --- */
const	struct_info_S*
structure_info		(const gb_struct_type_E	struct_type)
{
	register const	struct_info_S	*StrInf;
	register guint			i;
	
	/* fetch the widget information
	*/
	StrInf=NULL;
	for (i=0; i<structure_info_count; i++)
		if (structure_info_list[i]->struct_type==struct_type) {
			StrInf=structure_info_list[i];
			break;
		}
	
	g_assert(StrInf);
	
	return StrInf;
}


const	field_info_S*
field_info		(const struct_info_S	*StrInf,
			 const guint		field_num)
{
	g_assert(StrInf);
	
	g_assert(field_num<StrInf->field_count);
	
	return &StrInf->fields[field_num];
}


const	field_info_S*
field_info_by_struct_type	(const gb_struct_type_E	struct_type,
				 const guint		field_num)
{
	register const	struct_info_S	*StrInf;
	
	StrInf=structure_info(struct_type);
	
	g_assert(field_num<StrInf->field_count);
	
	return &StrInf->fields[field_num];
}


void
structure_field_set_default	(gb_any_S		*Struct,
				 const	field_info_S	*FldInf)
{
	register field_value_U		value;
	
	g_assert(GB_IS_STRUCT(Struct));
	g_assert(FldInf);
	
	switch (FldInf->field_type) {
	case	FIELD_FLOAT:
		value.float_val=FldInf->dfl_val;
		break;
	
	case	FIELD_BITS:
		value.bit_val=(unsigned)FldInf->dfl_val;
		break;
	
	case	FIELD_INT:
		value.int_val=FldInf->dfl_val;
		break;
	
	case	FIELD_ENUM:
		value.enum_val=(unsigned)FldInf->dfl_val;
		break;
	
	case	FIELD_BOOLEAN:
		value.bool_val=(unsigned)FldInf->dfl_val;
		break;

	case	FIELD_STRING:
		value.string=NULL;
		
		if (GB_IS_WIDDAT(Struct) && FIELD_FLAGS(FldInf)&FOPT_SYMBOL_NAME) {
			value.string=(gchar*)widget_data_symbol_name_get(GB_wCAST(base, Struct));
			g_assert(value.string);
			g_assert(strlen(value.string)>0);
		}
		break;
	
	case	FIELD_POINTER:
		value.string="NULL";
		break;
	
	case	FIELD_WD_LINK:
		value.link=NULL;
		break;
	
	default:
		g_assert_not_reached();
		break;
	}
	
	structure_field_set_value(Struct, value, FldInf);
}


gboolean
structure_field_has_default		(gb_any_S		*Struct,
					 const	field_info_S	*FldInf)
{
	register gboolean	is_default;
	
	g_assert(GB_IS_STRUCT(Struct));
	g_assert(FldInf);
	
	is_default=FALSE;
	switch (FldInf->field_type) {
	case	FIELD_FLOAT:
		if (FIELD_VALUE_U(Struct, FldInf->field_offset)->float_val==FldInf->dfl_val)
			is_default=TRUE;
		break;
	
	case	FIELD_BITS:
		if (FIELD_VALUE_U(Struct, FldInf->field_offset)->bit_val==
		    (unsigned)FldInf->dfl_val)
			is_default=TRUE;
		break;
	
	case	FIELD_INT:
		if (FIELD_VALUE_U(Struct, FldInf->field_offset)->int_val==
		    FldInf->dfl_val)
			is_default=TRUE;
		break;
	
	case	FIELD_ENUM:
		if (FIELD_VALUE_U(Struct, FldInf->field_offset)->enum_val==
		    (unsigned)FldInf->dfl_val)
			is_default=TRUE;
		break;
	
	case	FIELD_BOOLEAN:
		if (FIELD_VALUE_U(Struct, FldInf->field_offset)->bool_val==
		    (unsigned)FldInf->dfl_val)
			is_default=TRUE;
		break;

	case	FIELD_STRING:
		if (GB_IS_WIDDAT(Struct) && FIELD_FLAGS(FldInf)&FOPT_SYMBOL_NAME) {
			register const gchar	*string;
			
			string=structure_field_get_value_str(Struct, FldInf);
			
			if (strcmp(string, widget_data_symbol_name_get(GB_wCAST(base, Struct)))==0)
				is_default=TRUE;
		} else if (!FIELD_VALUE_U(Struct, FldInf->field_offset)->string ||
			   FIELD_VALUE_U(Struct, FldInf->field_offset)->string[0]==0)
			is_default=TRUE;
		break;
	
	case	FIELD_POINTER:
		g_assert(FIELD_VALUE_U(Struct, FldInf->field_offset)->string);
		if (strcmp(FIELD_VALUE_U(Struct, FldInf->field_offset)->string, "NULL")==0)
			is_default=TRUE;
		break;
	
	case	FIELD_WD_LINK:
		is_default=FIELD_VALUE_U(Struct, FldInf->field_offset)->link==NULL;
		break;
	
	default:
		g_assert_not_reached();
		break;
	}
	
	return is_default;
}


void
structure_field_set_value	(gb_any_S		*Struct,
				 field_value_U		value,
				 const	field_info_S	*FldInf)
{
	g_assert(GB_IS_STRUCT(Struct));
	g_assert(FldInf);
	
	switch (FldInf->field_type) {
	register field_value_U	dupval;
	
	case	FIELD_FLOAT:
		value.float_val=CLAMP(value.float_val, FldInf->minimum, FldInf->maximum);
		FIELD_VALUE_U(Struct, FldInf->field_offset)->float_val=value.float_val;
		break;
	
	case	FIELD_BITS:
		dupval.bit_val=FIELD_VALUE_U(Struct, FldInf->field_offset)->bit_val;
		dupval.bit_val &= ~(unsigned)FldInf->maximum;
		dupval.bit_val |= value.bit_val & (unsigned)FldInf->maximum;
		FIELD_VALUE_U(Struct, FldInf->field_offset)->bit_val=dupval.bit_val;
		break;
	
	case	FIELD_INT:
		value.int_val=CLAMP(value.int_val, FldInf->minimum, FldInf->maximum);
		FIELD_VALUE_U(Struct, FldInf->field_offset)->int_val=value.int_val;
		break;
	
	case	FIELD_ENUM:
		value.enum_val=CLAMP(value.enum_val, (unsigned)FldInf->minimum,
							(unsigned) FldInf->maximum);
		FIELD_VALUE_U(Struct, FldInf->field_offset)->enum_val=value.enum_val;
		break;
	
	case	FIELD_BOOLEAN:
		FIELD_VALUE_U(Struct, FldInf->field_offset)->bool_val=(value.bool_val!=0);
		break;
	
	case	FIELD_STRING:
		g_free(FIELD_VALUE_U(Struct, FldInf->field_offset)->string);
		FIELD_VALUE_U(Struct, FldInf->field_offset)->string=NULL;
		if (!value.string || value.string[0]==0)
			value.string=NULL;

		if (GB_IS_WIDDAT(Struct)) {
			register gb_wdat_base_S	*WidDat=GB_wCAST(base, Struct);
			
			if ( !value.string &&
			     (FIELD_FLAGS(FldInf)&FOPT_FORCE_DFL) &&
			     (FIELD_FLAGS(FldInf)&FOPT_SYMBOL_NAME) ) {
				value.string=(gchar*)widget_data_symbol_name_get(WidDat);
				g_assert(value.string);
				g_assert(strlen(value.string)>0);
			}
			if (value.string && FIELD_FLAGS(FldInf)&FOPT_STRING_IS_CHILD) {
				UPDATE_CHILD_COUNT(WidDat);
				if (HAS_MAX_CHILD_COUNT(WidDat))
					value.string=NULL;
			}
		}
		
		if (value.string)
			FIELD_VALUE_U(Struct, FldInf->field_offset)->string=g_strdup(value.string);
		else if (FIELD_FLAGS(FldInf)&FOPT_EMPTY_STRING)
			FIELD_VALUE_U(Struct, FldInf->field_offset)->string=g_strdup("");
		else
			FIELD_VALUE_U(Struct, FldInf->field_offset)->string=NULL;
		
		if (GB_IS_WIDDAT(Struct))
			UPDATE_CHILD_COUNT(GB_wCAST(base, Struct));
		break;
	
	case	FIELD_POINTER:
		g_free(FIELD_VALUE_U(Struct, FldInf->field_offset)->string);
		FIELD_VALUE_U(Struct, FldInf->field_offset)->string=NULL;
		if (!value.string || value.string[0]==0)
			value.string="NULL";
		FIELD_VALUE_U(Struct, FldInf->field_offset)->string=g_strdup(value.string);
		break;
	
	case	FIELD_WD_LINK:
		if (FIELD_VALUE_U(Struct, FldInf->field_offset)->link) {
			register gb_wdat_base_S	*LnkDat;
			
			LnkDat=FIELD_VALUE_U(Struct, FldInf->field_offset)->link;
			g_assert(GB_IS_WIDDAT(LnkDat));
			
			GUBI_DATA(LnkDat)->link_refs=g_list_remove(GUBI_DATA(LnkDat)->link_refs, Struct);
			FIELD_VALUE_U(Struct, FldInf->field_offset)->link=NULL;
		}
		if (value.link) {
			register gb_wdat_base_S	*LnkDat;
			
			LnkDat=value.link;
			g_assert(GB_IS_WIDDAT(LnkDat));
			
			GUBI_DATA(LnkDat)->link_refs=g_list_prepend(GUBI_DATA(LnkDat)->link_refs, Struct);
			FIELD_VALUE_U(Struct, FldInf->field_offset)->link=value.link;
		}
		break;
	
	default:
		g_assert_not_reached();
		break;
	}
}


const gchar*
structure_field_get_value_str		(gb_any_S		*Struct,
					 const	field_info_S	*FldInf)
{
	static	 gchar		*buffer=NULL;
	register field_value_U	value;
	
	value=structure_field_get_value(Struct, FldInf);
	
	g_free(buffer);
	buffer=NULL;
	
	switch (FldInf->field_type) {
		register guint			i;
		register enum_field_func	enum_func;
		
	case	FIELD_FLOAT:
		buffer=g_new(gchar, 64);
		i=sprintf(buffer, "%.6f", value.float_val);
		i--;
		while (buffer[i]=='0')
			buffer[i--]=0;
		break;
	
	case	FIELD_BITS:
		buffer=g_new(gchar, 64);
		sprintf(buffer, "%u", value.bit_val);
		break;
	
	case	FIELD_INT:
		buffer=g_new(gchar, 64);
		sprintf(buffer, "%d", value.int_val);
		break;
	
	case	FIELD_ENUM:
		g_assert((enum_func=(enum_field_func)FldInf->field_func));
		buffer=g_strdup( (*enum_func) (value.enum_val, FALSE)) ;
		break;
	
	case	FIELD_BOOLEAN:
		buffer=g_strdup( value.bool_val!=0 ? "TRUE" : "FALSE" );
		break;

	case	FIELD_STRING:
		if (value.string)
			buffer=g_strdup(value.string);
		else
			buffer=NULL;
		break;
	
	case	FIELD_POINTER:
		g_assert(value.string);
		buffer=g_strdup(value.string);
		break;
	
	case	FIELD_WD_LINK:
		buffer=value.link ? g_strdup(widget_data_symbol_name_get(value.link)) : g_strdup("");
		break;
	
	default:
		g_assert_not_reached();
		break;
	}
	
	return buffer;
}


field_value_U
structure_field_get_value		(gb_any_S		*Struct,
					 const	field_info_S	*FldInf)
{
	register field_value_U	value;
	
	g_assert(GB_IS_STRUCT(Struct));
	g_assert(FldInf);
	
	switch (FldInf->field_type) {
		
	case	FIELD_FLOAT:
		value.float_val=FIELD_VALUE_U(Struct, FldInf->field_offset)->float_val;
		break;
	
	case	FIELD_BITS:
		value.bit_val=FIELD_VALUE_U(Struct, FldInf->field_offset)->bit_val;
		break;
	
	case	FIELD_INT:
		value.int_val=FIELD_VALUE_U(Struct, FldInf->field_offset)->int_val;
		break;
	
	case	FIELD_ENUM:
		value.enum_val=FIELD_VALUE_U(Struct, FldInf->field_offset)->enum_val;
		break;
	
	case	FIELD_BOOLEAN:
		value.bool_val=FIELD_VALUE_U(Struct, FldInf->field_offset)->bool_val;
		break;

	case	FIELD_STRING:
	case	FIELD_POINTER:
		value.string=FIELD_VALUE_U(Struct, FldInf->field_offset)->string;
		break;
	
	case	FIELD_WD_LINK:
		value.link=FIELD_VALUE_U(Struct, FldInf->field_offset)->link;
		break;
	
	default:
		g_assert_not_reached();
		break;
	}
	
	return value;
}


guint
structure_field_bits_count		(const	field_info_S	*FldInf)
{
	register guint		count;
	register guint		i;
	register guint32	bits;
	
	g_assert(FldInf);
	g_assert(FldInf->field_type==FIELD_BITS);
	
	count=0;
	bits=(unsigned)FldInf->maximum;
	
	for (i=0; i<32; i++)
		if (bits & (1<<i))
			count++;
	
	return count;
}


guint32
structure_field_bit_value		(const	field_info_S	*FldInf,
					 guint			bit_num)
{
	register guint		count;
	register guint		i;
	register guint32	bits;
	
	g_assert(bit_num>0 && bit_num<=structure_field_bits_count(FldInf));
	
	count=0;
	bits=(unsigned)FldInf->maximum;
	
	for (i=0; i<32; i++) {
		if (bits & (1<<i))
			count++;
		if (count==bit_num)
			break;
	}
	
	return 1<<i;
}


void
structure_field_bit_set			(gb_any_S		*Struct,
					 const	field_info_S	*FldInf,
					 guint			bit_num,
					 gboolean		value)
{
	register guint32	bit_val;
	
	g_assert(GB_IS_STRUCT(Struct));
	g_assert((bit_val=structure_field_bit_value(FldInf, bit_num)));
	
	if (value)
		FIELD_VALUE_U(Struct, FldInf->field_offset)->bit_val|=bit_val;
	else
		FIELD_VALUE_U(Struct, FldInf->field_offset)->bit_val&=~bit_val;
}


void
structure_fields_initialize	(gb_any_S		*Struct,
				 const struct_info_S	*StrInf)
{
	register guint			i;
	
	g_assert(GB_IS_STRUCT(Struct));
	g_assert(StrInf);
	
	for (i=0; i<StrInf->field_count; i++) {
		register const	field_info_S	*FldInf;
		
		FldInf=field_info(StrInf, i);
		FIELD_VALUE_U(Struct, FldInf->field_offset)->string=NULL;
		structure_field_set_default(Struct, FldInf);
	}
}


void
structure_fields_free	(gb_any_S		*Struct,
			 const struct_info_S	*StrInf)
{
	register guint			i;
	
	g_assert(GB_IS_STRUCT(Struct));
	g_assert(StrInf);
	
	for (i=0; i<StrInf->field_count; i++) {
		register const	field_info_S	*FldInf;
		
		FldInf=field_info(StrInf, i);
		
		switch (FldInf->field_type) {
			register field_value_U	value;
		
		case	FIELD_BOOLEAN:
		case	FIELD_BITS:
		case	FIELD_FLOAT:
		case	FIELD_INT:
		case	FIELD_ENUM:
			break;
			
		case	FIELD_WD_LINK:
			value.link=NULL;
			structure_field_set_value(Struct, value, FldInf);
			break;
		
		case	FIELD_POINTER:
		case	FIELD_STRING:
			g_free(FIELD_VALUE_U(Struct, FldInf->field_offset)->string);
			FIELD_VALUE_U(Struct, FldInf->field_offset)->string=NULL;
			break;
		
		default:
			g_assert_not_reached();
			break;
		}
	}
}


guint
widget_data_base_child_count	(gb_wdat_base_S	*WidDat)
{
	register guint			i;
	register guint			child_count;
	register const struct_info_S	*StrInf;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	
	StrInf=structure_info(WidDat->type);
	child_count=0;
	
	for (i=0; i<StrInf->field_count; i++) {
		register const	field_info_S	*FldInf;
		
		FldInf=field_info(StrInf, i);
		
		if (FldInf->field_type==FIELD_STRING &&
		    FIELD_FLAGS(FldInf)&FOPT_STRING_IS_CHILD &&
		    structure_field_get_value(GB_CAST(any, WidDat), FldInf).string )
			child_count++;
	}
	
	if (child_count > structure_info(WidDat->type)->widget_child_count)
		g_error("%s\nencountered internal child_count oddity: %s: %d>%d",
			FUNCNAME,
			to_UpCase(structure_info(WidDat->type)->struct_name),
			child_count,
			structure_info(WidDat->type)->widget_child_count);

	return child_count;
}


void
widget_data_link_remove		(gb_wdat_base_S		*WidDat,
				 gb_wdat_base_S		*LinkDat)
{
	register const struct_info_S	*StrInf;
	register guint			i;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	g_assert(GB_IS_WIDDAT(LinkDat));
	
	StrInf=structure_info(WidDat->type);
	
	for (i=0; i<StrInf->field_count; i++) {
		register const field_info_S	*FldInf;
		
		FldInf=field_info(StrInf, i);
		
		if (FldInf->field_type!=FIELD_WD_LINK &&
		    FldInf->field_type!=FIELD_AUX_LINK)
			continue;
		
		if (structure_field_get_value(GB_CAST(any, WidDat), FldInf).link==LinkDat) {
			register field_value_U		value;
			
			value.link=NULL;
			structure_field_set_value(GB_CAST(any, WidDat), value, FldInf);
		}
	}
}
