--
-- This file is part of TALER
-- Copyright (C) 2014--2023 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--

CREATE FUNCTION create_table_reserves_open_deposits(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'reserves_open_deposits';
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(reserve_open_deposit_uuid BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',reserve_sig BYTEA NOT NULL CHECK (LENGTH(reserve_sig)=64)'
      ',reserve_pub BYTEA NOT NULL CHECK (LENGTH(reserve_pub)=32)'
      ',coin_pub BYTEA NOT NULL CHECK (LENGTH(coin_pub)=32)'
      ',coin_sig BYTEA NOT NULL CHECK (LENGTH(coin_sig)=64)'
      ',contribution taler_amount NOT NULL'
    ') %s ;'
    ,table_name
    ,'PARTITION BY HASH (coin_pub)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'coin contributions paying for a reserve to remain open'
    ,table_name
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'Identifies the specific reserve being paid for (possibly together with reserve_sig).'
    ,'reserve_pub'
    ,table_name
    ,partition_suffix
  );
END
$$;


CREATE FUNCTION constrain_table_reserves_open_deposits(
  IN partition_suffix TEXT
)
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT default 'reserves_open_deposits';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);
  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name || ' '
      'ADD CONSTRAINT ' || table_name || '_coin_unique '
        'PRIMARY KEY (coin_pub,coin_sig)'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_uuid '
    'ON ' || table_name || ' '
    '(reserve_open_deposit_uuid);'
  );
  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_reserve '
    'ON ' || table_name || ' '
    '(reserve_pub);'
  );
END
$$;


CREATE OR REPLACE FUNCTION reserves_open_deposits_insert_trigger()
  RETURNS trigger
  LANGUAGE plpgsql
  AS $$
BEGIN
  INSERT INTO exchange.coin_history
    (coin_pub
    ,table_name
    ,serial_id)
 VALUES
     (NEW.coin_pub
    ,'reserves_open_deposits'
    ,NEW.reserve_open_deposit_uuid);
  RETURN NEW;
END $$;
COMMENT ON FUNCTION reserves_open_deposits_insert_trigger()
  IS 'Automatically generate coin history entry.';


CREATE FUNCTION master_table_reserves_open_deposits()
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  CREATE TRIGGER reserves_open_deposits_on_insert
    AFTER INSERT
     ON reserves_open_deposits
     FOR EACH ROW EXECUTE FUNCTION reserves_open_deposits_insert_trigger();
END $$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('reserves_open_deposits'
    ,'exchange-0002'
    ,'create'
    ,TRUE
    ,FALSE),
    ('reserves_open_deposits'
    ,'exchange-0002'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('reserves_open_deposits'
    ,'exchange-0002'
    ,'master'
    ,TRUE
    ,FALSE);
