#! /usr/bin/env python
# -*- coding: iso-8859-1 -*-
""" crypto.cipher.aes_test

    Tests for AES encryption algorithm

    Copyright  (c) 2002 by Paul A. Lambert
    Read LICENSE.txt for license information.
"""
import unittest
from crypto.cipher.aes  import AES
from crypto.cipher.base import noPadding
from binascii           import a2b_hex, b2a_hex

class AES_TestVectors(unittest.TestCase):
    """ Test AES algorithm using know values."""
    def testAES_KAT(self):
        """ Run AES know answer tests from hex input """
        def AEStestVector(i, key, pt, ct):
            """ Test vectors provide test name(i), key, plain text (pt) and know cipher text (ct) """
            bkey, plainText, knownCipherText = a2b_hex(key), a2b_hex(pt), a2b_hex(ct)
            kSize = len(bkey)
            alg = AES(bkey, keySize=kSize, padding=noPadding())
            self.assertEqual( alg.encrypt(plainText),       knownCipherText )
            self.assertEqual( alg.decrypt(knownCipherText), plainText )

        """ AES tests using FIPS example vectors """

        AEStestVector( i   = 'NIST Example',
                       key = '2b7e151628aed2a6abf7158809cf4f3c',
                       pt  = '3243f6a8885a308d313198a2e0370734',
                       ct  = '3925841d02dc09fbdc118597196a0b32')
        AEStestVector( i   = 'NIST 128 bit encrypt decrypt example',
                       key = '000102030405060708090a0b0c0d0e0f',
                       pt  = '00112233445566778899aabbccddeeff',
                       ct  = '69c4e0d86a7b0430d8cdb78070b4c55a')

        """ AES Known Answer Tests with Variable Text (block size only 128) from ecb_vt.txt"""

        AEStestVector( i   = 'aes-ecb-vt-128-1',
                       key = '00000000000000000000000000000000',
                       pt  = '80000000000000000000000000000000',
                       ct  = '3AD78E726C1EC02B7EBFE92B23D9EC34')
        AEStestVector( i   = 'aes-ecb-vt-128-2',
                       key = '00000000000000000000000000000000',
                       pt  = '40000000000000000000000000000000',
                       ct  = '45BC707D29E8204D88DFBA2F0B0CAD9B')
        AEStestVector( i   = 'aes-ecb-vt-128-3',
                       key = '00000000000000000000000000000000',
                       pt  = '20000000000000000000000000000000',
                       ct  = '161556838018F52805CDBD6202002E3F')
        AEStestVector( i   = 'aes-ecb-vt-128-4',
                       key = '00000000000000000000000000000000',
                       pt  = '10000000000000000000000000000000',
                       ct  = 'F5569B3AB6A6D11EFDE1BF0A64C6854A')
        AEStestVector( i   = 'aes-ecb-vt-128-5',
                       key = '00000000000000000000000000000000',
                       pt  = '08000000000000000000000000000000',
                       ct  = '64E82B50E501FBD7DD4116921159B83E')
        AEStestVector( i   = 'aes-ecb-vt-128-6',
                       key = '00000000000000000000000000000000',
                       pt  = '04000000000000000000000000000000',
                       ct  = 'BAAC12FB613A7DE11450375C74034041')
        AEStestVector( i   = 'aes-ecb-vt-128-7',
                       key = '00000000000000000000000000000000',
                       pt  = '02000000000000000000000000000000',
                       ct  = 'BCF176A7EAAD8085EBACEA362462A281')
        AEStestVector( i   = 'aes-ecb-vt-128-8',
                       key = '00000000000000000000000000000000',
                       pt  = '01000000000000000000000000000000',
                       ct  = '47711816E91D6FF059BBBF2BF58E0FD3')
        AEStestVector( i   = 'aes-ecb-vt-128-9',
                       key = '00000000000000000000000000000000',
                       pt  = '00800000000000000000000000000000',
                       ct  = 'B970DFBE40698AF1638FE38BD3DF3B2F')
        AEStestVector( i   = 'aes-ecb-vt-128-10',
                       key = '00000000000000000000000000000000',
                       pt  = '00400000000000000000000000000000',
                       ct  = 'F95B59A44F391E14CF20B74BDC32FCFF')
        AEStestVector( i   = 'aes-ecb-vt-128-11',
                       key = '00000000000000000000000000000000',
                       pt  = '00200000000000000000000000000000',
                       ct  = '720F74AE04A2A435B9A7256E49378F5B')
        AEStestVector( i   = 'aes-ecb-vt-128-12',
                       key = '00000000000000000000000000000000',
                       pt  = '00100000000000000000000000000000',
                       ct  = '2A0445F61D36BFA7E277070730CF76DA')
        AEStestVector( i   = 'aes-ecb-vt-128-13',
                       key = '00000000000000000000000000000000',
                       pt  = '00080000000000000000000000000000',
                       ct  = '8D0536B997AEFEC1D94011BAB6699A03')
        AEStestVector( i   = 'aes-ecb-vt-128-14',
                       key = '00000000000000000000000000000000',
                       pt  = '00040000000000000000000000000000',
                       ct  = '674F002E19F6ED47EFF319E51FAD4498')
        AEStestVector( i   = 'aes-ecb-vt-128-15',
                       key = '00000000000000000000000000000000',
                       pt  = '00020000000000000000000000000000',
                       ct  = '292C02C5CB9163C80AC0F6CF1DD8E92D')
        AEStestVector( i   = 'aes-ecb-vt-128-16',
                       key = '00000000000000000000000000000000',
                       pt  = '00010000000000000000000000000000',
                       ct  = 'FA321CF18EF5FE727DD82A5C1E945141')
        AEStestVector( i   = 'aes-ecb-vt-128-17',
                       key = '00000000000000000000000000000000',
                       pt  = '00008000000000000000000000000000',
                       ct  = 'A5A7AFE1034C39CCCEBE3C584BC0BE05')
        AEStestVector( i   = 'aes-ecb-vt-128-18',
                       key = '00000000000000000000000000000000',
                       pt  = '00004000000000000000000000000000',
                       ct  = '4FF5A52E697E77D081205DBDB21CEA39')
        AEStestVector( i   = 'aes-ecb-vt-128-19',
                       key = '00000000000000000000000000000000',
                       pt  = '00002000000000000000000000000000',
                       ct  = '209E88DC94C9003000CE0769AF7B7166')
        AEStestVector( i   = 'aes-ecb-vt-128-20',
                       key = '00000000000000000000000000000000',
                       pt  = '00001000000000000000000000000000',
                       ct  = '5DEE41AF864CB4B650E5F51551824D38')
        AEStestVector( i   = 'aes-ecb-vt-128-21',
                       key = '00000000000000000000000000000000',
                       pt  = '00000800000000000000000000000000',
                       ct  = 'A79A63FA7E4503AE6D6E09F5F9053030')
        AEStestVector( i   = 'aes-ecb-vt-128-22',
                       key = '00000000000000000000000000000000',
                       pt  = '00000400000000000000000000000000',
                       ct  = 'A48316749FAE7FAC7002031A6AFD8BA7')
        AEStestVector( i   = 'aes-ecb-vt-128-23',
                       key = '00000000000000000000000000000000',
                       pt  = '00000200000000000000000000000000',
                       ct  = 'D6EEE8A7357A0E1D64262CA9C337AC42')
        AEStestVector( i   = 'aes-ecb-vt-128-24',
                       key = '00000000000000000000000000000000',
                       pt  = '00000100000000000000000000000000',
                       ct  = 'B013CA8A62A858053E9FB667ED39829E')
        AEStestVector( i   = 'aes-ecb-vt-128-25',
                       key = '00000000000000000000000000000000',
                       pt  = '00000080000000000000000000000000',
                       ct  = 'DF6EA9E4538A45A52D5C1A43C88F4B55')
        AEStestVector( i   = 'aes-ecb-vt-128-26',
                       key = '00000000000000000000000000000000',
                       pt  = '00000040000000000000000000000000',
                       ct  = '7D03BA451371591D3FD5547D9165C73B')
        AEStestVector( i   = 'aes-ecb-vt-128-27',
                       key = '00000000000000000000000000000000',
                       pt  = '00000020000000000000000000000000',
                       ct  = '0E0426281A6277E186499D365D5F49FF')
        AEStestVector( i   = 'aes-ecb-vt-128-28',
                       key = '00000000000000000000000000000000',
                       pt  = '00000010000000000000000000000000',
                       ct  = 'DBC02169DD2059E6CC4C57C1FEDF5AB4')
        AEStestVector( i   = 'aes-ecb-vt-128-29',
                       key = '00000000000000000000000000000000',
                       pt  = '00000008000000000000000000000000',
                       ct  = '826590E05D167DA6F00DCC75E22788EB')
        AEStestVector( i   = 'aes-ecb-vt-128-30',
                       key = '00000000000000000000000000000000',
                       pt  = '00000004000000000000000000000000',
                       ct  = '34A73F21A04421D9786335FAAB49423A')
        AEStestVector( i   = 'aes-ecb-vt-128-31',
                       key = '00000000000000000000000000000000',
                       pt  = '00000002000000000000000000000000',
                       ct  = 'ED347D0E0128EE1A7392A1D36AB78AA9')
        AEStestVector( i   = 'aes-ecb-vt-128-32',
                       key = '00000000000000000000000000000000',
                       pt  = '00000001000000000000000000000000',
                       ct  = 'EE944B2FE6E9FC888042608DA9615F75')
        AEStestVector( i   = 'aes-ecb-vt-128-33',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000800000000000000000000000',
                       ct  = '9E7C85A909EF7218BA7947CFB4718F46')
        AEStestVector( i   = 'aes-ecb-vt-128-34',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000400000000000000000000000',
                       ct  = '811AE07A0B2B1F816587FA73699AE77D')
        AEStestVector( i   = 'aes-ecb-vt-128-35',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000200000000000000000000000',
                       ct  = '68466FBF43C2FE13D4B18F7EC5EA745F')
        AEStestVector( i   = 'aes-ecb-vt-128-36',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000100000000000000000000000',
                       ct  = 'D20B015C7191B219780956E6101F9354')
        AEStestVector( i   = 'aes-ecb-vt-128-37',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000080000000000000000000000',
                       ct  = '5939D5C1BBF54EE1B3E326D757BDDE25')
        AEStestVector( i   = 'aes-ecb-vt-128-38',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000040000000000000000000000',
                       ct  = 'B1FDAFE9A0240E8FFEA19CE94B5105D3')
        AEStestVector( i   = 'aes-ecb-vt-128-39',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000020000000000000000000000',
                       ct  = 'D62962ECE02CDD68C06BDFEFB2F9495B')
        AEStestVector( i   = 'aes-ecb-vt-128-40',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000010000000000000000000000',
                       ct  = 'B3BB2DE6F3C26587BA8BAC4F7AD9499A')
        AEStestVector( i   = 'aes-ecb-vt-128-41',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000008000000000000000000000',
                       ct  = 'E0B1072D6D9FF703D6FBEF77852B0A6B')
        AEStestVector( i   = 'aes-ecb-vt-128-42',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000004000000000000000000000',
                       ct  = 'D8DD51C907F478DE0228E83E61FD1758')
        AEStestVector( i   = 'aes-ecb-vt-128-43',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000002000000000000000000000',
                       ct  = 'A42DFFE6E7C1671C06A25236FDD10017')
        AEStestVector( i   = 'aes-ecb-vt-128-44',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000001000000000000000000000',
                       ct  = '25ACF141550BFAB9EF451B6C6A5B2163')
        AEStestVector( i   = 'aes-ecb-vt-128-45',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000800000000000000000000',
                       ct  = '4DA7FCA3949B16E821DBC84F19581018')
        AEStestVector( i   = 'aes-ecb-vt-128-46',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000400000000000000000000',
                       ct  = '7D49B6347CBCC8919C7FA96A37A7A215')
        AEStestVector( i   = 'aes-ecb-vt-128-47',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000200000000000000000000',
                       ct  = '900024B29A08C6721B95BA3B753DDB4D')
        AEStestVector( i   = 'aes-ecb-vt-128-48',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000100000000000000000000',
                       ct  = '6D2182FB283B6934D90BA7848CAB5E66')
        AEStestVector( i   = 'aes-ecb-vt-128-49',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000080000000000000000000',
                       ct  = 'F73EF01B448D23A4D90DE8B2F9666E7A')
        AEStestVector( i   = 'aes-ecb-vt-128-50',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000040000000000000000000',
                       ct  = '4AD9CDA2418643E9A3D926AF5E6B0412')
        AEStestVector( i   = 'aes-ecb-vt-128-51',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000020000000000000000000',
                       ct  = '7CAEC8E7E5953997D545B033201C8C5B')
        AEStestVector( i   = 'aes-ecb-vt-128-52',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000010000000000000000000',
                       ct  = '3C43CA1F6B6864503E27B48D88230CF5')
        AEStestVector( i   = 'aes-ecb-vt-128-53',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000008000000000000000000',
                       ct  = '44F779B93108FE9FEEC880D79BA74488')
        AEStestVector( i   = 'aes-ecb-vt-128-54',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000004000000000000000000',
                       ct  = '9E50E8D9CFD3A682A78E527C9072A1CF')
        AEStestVector( i   = 'aes-ecb-vt-128-55',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000002000000000000000000',
                       ct  = '68D000CBC838BBE3C505D6F814C01F28')
        AEStestVector( i   = 'aes-ecb-vt-128-56',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000001000000000000000000',
                       ct  = '2CB2A9FEC1ACD1D9B0FA05205E304F57')
        AEStestVector( i   = 'aes-ecb-vt-128-57',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000800000000000000000',
                       ct  = '01EB2806606E46444520A5CC6180CD4B')
        AEStestVector( i   = 'aes-ecb-vt-128-58',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000400000000000000000',
                       ct  = 'DAA9B25168CC702326F217F1A0C0B162')
        AEStestVector( i   = 'aes-ecb-vt-128-59',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000200000000000000000',
                       ct  = '3E07E648975D9578D03555B1755807ED')
        AEStestVector( i   = 'aes-ecb-vt-128-60',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000100000000000000000',
                       ct  = '0B45F52E802C8B8DE09579425B80B711')
        AEStestVector( i   = 'aes-ecb-vt-128-61',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000080000000000000000',
                       ct  = '659595DA0B68F6DF0DD6CA77202986E1')
        AEStestVector( i   = 'aes-ecb-vt-128-62',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000040000000000000000',
                       ct  = '05FF42873893536E58C8FA98A45C73C4')
        AEStestVector( i   = 'aes-ecb-vt-128-63',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000020000000000000000',
                       ct  = 'B5B03421DE8BBFFC4EADEC767339A9BD')
        AEStestVector( i   = 'aes-ecb-vt-128-64',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000010000000000000000',
                       ct  = '788BCD111ECF73D4E78D2E21BEF55460')
        AEStestVector( i   = 'aes-ecb-vt-128-65',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000008000000000000000',
                       ct  = '909CD9EC6790359F982DC6F2393D5315')
        AEStestVector( i   = 'aes-ecb-vt-128-66',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000004000000000000000',
                       ct  = '332950F361535FF24EFAC8C76293F12C')
        AEStestVector( i   = 'aes-ecb-vt-128-67',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000002000000000000000',
                       ct  = 'A68CCD4E330FFDA9D576DA436DB53D75')
        AEStestVector( i   = 'aes-ecb-vt-128-68',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000001000000000000000',
                       ct  = '27C8A1CCFDB0B015D1ED5B3E77143791')
        AEStestVector( i   = 'aes-ecb-vt-128-69',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000800000000000000',
                       ct  = 'D76A4B95887A77DF610DD3E1D3B20325')
        AEStestVector( i   = 'aes-ecb-vt-128-70',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000400000000000000',
                       ct  = 'C068AB0DE71C66DAE83C361EF4B2D989')
        AEStestVector( i   = 'aes-ecb-vt-128-71',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000200000000000000',
                       ct  = 'C2120BCD49EDA9A288B3B4BE79AC8158')
        AEStestVector( i   = 'aes-ecb-vt-128-72',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000100000000000000',
                       ct  = '0C546F62BF2773CD0F564FCECA7BA688')
        AEStestVector( i   = 'aes-ecb-vt-128-73',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000080000000000000',
                       ct  = '18F3462BEDE4920213CCB66DAB1640AA')
        AEStestVector( i   = 'aes-ecb-vt-128-74',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000040000000000000',
                       ct  = 'FE42F245EDD0E24B216AEBD8B392D690')
        AEStestVector( i   = 'aes-ecb-vt-128-75',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000020000000000000',
                       ct  = '3D3EEBC8D3D1558A194C2D00C337FF2B')
        AEStestVector( i   = 'aes-ecb-vt-128-76',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000010000000000000',
                       ct  = '29AAEDF043E785DB42836F79BE6CBA28')
        AEStestVector( i   = 'aes-ecb-vt-128-77',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000008000000000000',
                       ct  = '215F90C6744E2944358E78619159A611')
        AEStestVector( i   = 'aes-ecb-vt-128-78',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000004000000000000',
                       ct  = '8606B1AA9E1D548E5442B06551E2C6DC')
        AEStestVector( i   = 'aes-ecb-vt-128-79',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000002000000000000',
                       ct  = '987BB4B8740EC0EDE7FEA97DF033B5B1')
        AEStestVector( i   = 'aes-ecb-vt-128-80',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000001000000000000',
                       ct  = 'C0A3500DA5B0AE07D2F450930BEEDF1B')
        AEStestVector( i   = 'aes-ecb-vt-128-81',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000800000000000',
                       ct  = '525FDF8312FE8F32C781481A8DAAAE37')
        AEStestVector( i   = 'aes-ecb-vt-128-82',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000400000000000',
                       ct  = 'BFD2C56AE5FB9C9DE33A6944572A6487')
        AEStestVector( i   = 'aes-ecb-vt-128-83',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000200000000000',
                       ct  = '7975A57A425CDF5AA1FA929101F650B0')
        AEStestVector( i   = 'aes-ecb-vt-128-84',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000100000000000',
                       ct  = 'BF174BC49609A8709B2CD8366DAA79FE')
        AEStestVector( i   = 'aes-ecb-vt-128-85',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000080000000000',
                       ct  = '06C50C43222F56C874B1704E9F44BF7D')
        AEStestVector( i   = 'aes-ecb-vt-128-86',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000040000000000',
                       ct  = '0CEC48CD34043EA29CA3B8ED5278721E')
        AEStestVector( i   = 'aes-ecb-vt-128-87',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000020000000000',
                       ct  = '9548EA34A1560197B304D0ACB8A1698D')
        AEStestVector( i   = 'aes-ecb-vt-128-88',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000010000000000',
                       ct  = '22F9E9B1BD73B6B5B7D3062C986272F3')
        AEStestVector( i   = 'aes-ecb-vt-128-89',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000008000000000',
                       ct  = 'FEE8E934BD0873295059002230E298D4')
        AEStestVector( i   = 'aes-ecb-vt-128-90',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000004000000000',
                       ct  = '1B08E2E3EB820D139CB4ABBDBE81D00D')
        AEStestVector( i   = 'aes-ecb-vt-128-91',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000002000000000',
                       ct  = '0021177681E4D90CEAF69DCED0145125')
        AEStestVector( i   = 'aes-ecb-vt-128-92',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000001000000000',
                       ct  = '4A8E314452CA8A8A3619FC54BC423643')
        AEStestVector( i   = 'aes-ecb-vt-128-93',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000800000000',
                       ct  = '65047474F7222C94C6965425FF1BFD0A')
        AEStestVector( i   = 'aes-ecb-vt-128-94',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000400000000',
                       ct  = 'E123F551A9C4A8489622B16F961A9AA4')
        AEStestVector( i   = 'aes-ecb-vt-128-95',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000200000000',
                       ct  = 'EF05530948B80915028BB2B6FE429380')
        AEStestVector( i   = 'aes-ecb-vt-128-96',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000100000000',
                       ct  = '72535B7FE0F0F777CEDCD55CD77E2DDF')
        AEStestVector( i   = 'aes-ecb-vt-128-97',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000080000000',
                       ct  = '3423D8EFC31FA2F4C365C77D8F3B5C63')
        AEStestVector( i   = 'aes-ecb-vt-128-98',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000040000000',
                       ct  = 'DE0E51C264663F3C5DBC59580A98D8E4')
        AEStestVector( i   = 'aes-ecb-vt-128-99',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000020000000',
                       ct  = 'B2D9391166680947AB09264156719679')
        AEStestVector( i   = 'aes-ecb-vt-128-100',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000010000000',
                       ct  = '10DB79F23B06D263835C424AF749ADB7')
        AEStestVector( i   = 'aes-ecb-vt-128-101',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000008000000',
                       ct  = 'DDF72D27E6B01EC107EA3E005B59563B')
        AEStestVector( i   = 'aes-ecb-vt-128-102',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000004000000',
                       ct  = '8266B57485A5954A4236751DE07F6694')
        AEStestVector( i   = 'aes-ecb-vt-128-103',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000002000000',
                       ct  = '669A501E1F1ADE6E5523DE01D6DBC987')
        AEStestVector( i   = 'aes-ecb-vt-128-104',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000001000000',
                       ct  = 'C20C48F2989725D461D1DB589DC0896E')
        AEStestVector( i   = 'aes-ecb-vt-128-105',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000800000',
                       ct  = 'DE35158E7810ED1191825D2AA98FA97D')
        AEStestVector( i   = 'aes-ecb-vt-128-106',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000400000',
                       ct  = '4FE294F2C0F34D0671B693A237EBDDC8')
        AEStestVector( i   = 'aes-ecb-vt-128-107',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000200000',
                       ct  = '087AE74B10CCBFDF6739FEB9559C01A4')
        AEStestVector( i   = 'aes-ecb-vt-128-108',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000100000',
                       ct  = '5DC278970B7DEF77A5536C77AB59C207')
        AEStestVector( i   = 'aes-ecb-vt-128-109',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000080000',
                       ct  = '7607F078C77085184EAA9B060C1FBFFF')
        AEStestVector( i   = 'aes-ecb-vt-128-110',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000040000',
                       ct  = '9DB841531BCBE7998DAD19993FB3CC00')
        AEStestVector( i   = 'aes-ecb-vt-128-111',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000020000',
                       ct  = 'D6A089B654854A94560BAE13298835B8')
        AEStestVector( i   = 'aes-ecb-vt-128-112',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000010000',
                       ct  = 'E1E223C4CF90CC5D195B370D65114622')
        AEStestVector( i   = 'aes-ecb-vt-128-113',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000008000',
                       ct  = '1CBED73C50D053BDAD372CEEE54836A1')
        AEStestVector( i   = 'aes-ecb-vt-128-114',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000004000',
                       ct  = 'D309E69376D257ADF2BFDA152B26555F')
        AEStestVector( i   = 'aes-ecb-vt-128-115',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000002000',
                       ct  = '740F7649117F0DEE6EAA7789A9994C36')
        AEStestVector( i   = 'aes-ecb-vt-128-116',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000001000',
                       ct  = '76AE64417C297184D668C5FD908B3CE5')
        AEStestVector( i   = 'aes-ecb-vt-128-117',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000000800',
                       ct  = '6095FEA4AA8035591F1787A819C48787')
        AEStestVector( i   = 'aes-ecb-vt-128-118',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000000400',
                       ct  = 'D1FF4E7ACD1C79967FEBAB0F7465D450')
        AEStestVector( i   = 'aes-ecb-vt-128-119',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000000200',
                       ct  = '5F5AD3C42B9489557BB63BF49ECF5F8A')
        AEStestVector( i   = 'aes-ecb-vt-128-120',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000000100',
                       ct  = 'FB56CC09B680B1D07C5A52149E29F07C')
        AEStestVector( i   = 'aes-ecb-vt-128-121',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000000080',
                       ct  = 'FF49B8DF4A97CBE03833E66197620DAD')
        AEStestVector( i   = 'aes-ecb-vt-128-122',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000000040',
                       ct  = '5E070ADE533D2E090ED0F5BE13BC0983')
        AEStestVector( i   = 'aes-ecb-vt-128-123',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000000020',
                       ct  = '3AB4FB1D2B7BA376590A2C241D1F508D')
        AEStestVector( i   = 'aes-ecb-vt-128-124',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000000010',
                       ct  = '58B2431BC0BEDE02550F40238969EC78')
        AEStestVector( i   = 'aes-ecb-vt-128-125',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000000008',
                       ct  = '0253786E126504F0DAB90C48A30321DE')
        AEStestVector( i   = 'aes-ecb-vt-128-126',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000000004',
                       ct  = '200211214E7394DA2089B6ACD093ABE0')
        AEStestVector( i   = 'aes-ecb-vt-128-127',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000000002',
                       ct  = '0388DACE60B6A392F328C2B971B2FE78')
        AEStestVector( i   = 'aes-ecb-vt-128-128',
                       key = '00000000000000000000000000000000',
                       pt  = '00000000000000000000000000000001',
                       ct  = '58E2FCCEFA7E3061367F1D57A4E7455A')
        AEStestVector( i   = 'aes-ecb-vt-192-1',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '80000000000000000000000000000000',
                       ct  = '6CD02513E8D4DC986B4AFE087A60BD0C')
        AEStestVector( i   = 'aes-ecb-vt-192-2',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '40000000000000000000000000000000',
                       ct  = '423D2772A0CA56DAABB48D2129062987')
        AEStestVector( i   = 'aes-ecb-vt-192-3',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '20000000000000000000000000000000',
                       ct  = '1021F2A8DA70EB2219DC16804445FF98')
        AEStestVector( i   = 'aes-ecb-vt-192-4',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '10000000000000000000000000000000',
                       ct  = 'C636E35B402577F96974D8804295EBB8')
        AEStestVector( i   = 'aes-ecb-vt-192-5',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '08000000000000000000000000000000',
                       ct  = '1566D2E57E8393C19E29F892EA28A9A7')
        AEStestVector( i   = 'aes-ecb-vt-192-6',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '04000000000000000000000000000000',
                       ct  = '883C878FED70B36CC09D040F9619DD19')
        AEStestVector( i   = 'aes-ecb-vt-192-7',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '02000000000000000000000000000000',
                       ct  = '06734593A974965790E715594FC34AA9')
        AEStestVector( i   = 'aes-ecb-vt-192-8',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '01000000000000000000000000000000',
                       ct  = 'F19B389948D9A45534E5BD36C984134A')
        AEStestVector( i   = 'aes-ecb-vt-192-9',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00800000000000000000000000000000',
                       ct  = 'D8410DFC14FA6D175EC968EA8CAC514C')
        AEStestVector( i   = 'aes-ecb-vt-192-10',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00400000000000000000000000000000',
                       ct  = '7E6C6EBB4029A177CF7B2FDD9AC6BB7A')
        AEStestVector( i   = 'aes-ecb-vt-192-11',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00200000000000000000000000000000',
                       ct  = '4B51DD4850DC0A6C3A46D924003D2C27')
        AEStestVector( i   = 'aes-ecb-vt-192-12',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00100000000000000000000000000000',
                       ct  = '2E510A9D917B15BE32A192B12A668F23')
        AEStestVector( i   = 'aes-ecb-vt-192-13',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00080000000000000000000000000000',
                       ct  = '88F6F79962B0FB77FEA8E7C632D3108E')
        AEStestVector( i   = 'aes-ecb-vt-192-14',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00040000000000000000000000000000',
                       ct  = 'A3A35AB1D88DAF07B52794A0F065383A')
        AEStestVector( i   = 'aes-ecb-vt-192-15',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00020000000000000000000000000000',
                       ct  = 'DC6CC878433E2B3BB193049A4ECBFC53')
        AEStestVector( i   = 'aes-ecb-vt-192-16',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00010000000000000000000000000000',
                       ct  = 'EFCD3763EB7B1A415938248A9A5B4FD5')
        AEStestVector( i   = 'aes-ecb-vt-192-17',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00008000000000000000000000000000',
                       ct  = 'AB7E9FB9A66DBE5BB44854F07D9015EE')
        AEStestVector( i   = 'aes-ecb-vt-192-18',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00004000000000000000000000000000',
                       ct  = '8B8E9D3365F8F6743ECF7E33E99255A4')
        AEStestVector( i   = 'aes-ecb-vt-192-19',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00002000000000000000000000000000',
                       ct  = '54D37B4F176FF3D8F6AFC866066D8572')
        AEStestVector( i   = 'aes-ecb-vt-192-20',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00001000000000000000000000000000',
                       ct  = 'E83310889480FBF3C00342E3126D0D02')
        AEStestVector( i   = 'aes-ecb-vt-192-21',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000800000000000000000000000000',
                       ct  = 'D321AB2511F92F098174AA2DE6E85DA2')
        AEStestVector( i   = 'aes-ecb-vt-192-22',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000400000000000000000000000000',
                       ct  = 'D8E3F40B1112D5149D58C481DFA9983F')
        AEStestVector( i   = 'aes-ecb-vt-192-23',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000200000000000000000000000000',
                       ct  = '2454C4E0806639DDF19854D6C68054AD')
        AEStestVector( i   = 'aes-ecb-vt-192-24',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000100000000000000000000000000',
                       ct  = 'A5506D410F7CA32F3955DD79D9D09418')
        AEStestVector( i   = 'aes-ecb-vt-192-25',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000080000000000000000000000000',
                       ct  = '7908EE40677699568A7DC1AA317C7E4E')
        AEStestVector( i   = 'aes-ecb-vt-192-26',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000040000000000000000000000000',
                       ct  = 'B4B7B29DD43B2F5CF765E25192273982')
        AEStestVector( i   = 'aes-ecb-vt-192-27',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000020000000000000000000000000',
                       ct  = '92AFE9668159BEFFE2A86F8503260164')
        AEStestVector( i   = 'aes-ecb-vt-192-28',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000010000000000000000000000000',
                       ct  = '5C36A232FBA6D187A84657AD4028B18F')
        AEStestVector( i   = 'aes-ecb-vt-192-29',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000008000000000000000000000000',
                       ct  = 'A2E994DFAB3A798DF8F54F6DA87E58E2')
        AEStestVector( i   = 'aes-ecb-vt-192-30',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000004000000000000000000000000',
                       ct  = '6CDAB10A72ADF77D71D0765BAAE95631')
        AEStestVector( i   = 'aes-ecb-vt-192-31',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000002000000000000000000000000',
                       ct  = '9FE3C801BCAAF7BB800F2E6BF3278E21')
        AEStestVector( i   = 'aes-ecb-vt-192-32',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000001000000000000000000000000',
                       ct  = 'B459D90D9A6C392E5493BC91CF5A0863')
        AEStestVector( i   = 'aes-ecb-vt-192-33',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000800000000000000000000000',
                       ct  = '0518A9FA5007F6787E0FB4E5AC27D758')
        AEStestVector( i   = 'aes-ecb-vt-192-34',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000400000000000000000000000',
                       ct  = 'BED9795415D28599700ED7952384A963')
        AEStestVector( i   = 'aes-ecb-vt-192-35',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000200000000000000000000000',
                       ct  = 'F0140421173D60251EF6CAB0229B1B50')
        AEStestVector( i   = 'aes-ecb-vt-192-36',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000100000000000000000000000',
                       ct  = '460EB4652B3F6779EA28CB11B37529ED')
        AEStestVector( i   = 'aes-ecb-vt-192-37',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000080000000000000000000000',
                       ct  = 'C4283D351C960A6AC13CD19CCF03AE38')
        AEStestVector( i   = 'aes-ecb-vt-192-38',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000040000000000000000000000',
                       ct  = '6815A10047B2C834A798EBDCC6786C75')
        AEStestVector( i   = 'aes-ecb-vt-192-39',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000020000000000000000000000',
                       ct  = '99BA19F0CDD5990D0386B32CE56C9C4C')
        AEStestVector( i   = 'aes-ecb-vt-192-40',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000010000000000000000000000',
                       ct  = 'DE76F62C61E07915162DA13E79679DEC')
        AEStestVector( i   = 'aes-ecb-vt-192-41',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000008000000000000000000000',
                       ct  = 'DD0325D6854803D06D1D2277D5FB8D67')
        AEStestVector( i   = 'aes-ecb-vt-192-42',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000004000000000000000000000',
                       ct  = '580B71A41DE37D6FAC83CCB0B3BB1C97')
        AEStestVector( i   = 'aes-ecb-vt-192-43',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000002000000000000000000000',
                       ct  = 'E9B1AB470A1B02EF0FF5E6754A092C96')
        AEStestVector( i   = 'aes-ecb-vt-192-44',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000001000000000000000000000',
                       ct  = '8590620F5AF5993B7410282F4126BC1F')
        AEStestVector( i   = 'aes-ecb-vt-192-45',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000800000000000000000000',
                       ct  = '8D4914D2F1B22B2E268E66E532D29D7C')
        AEStestVector( i   = 'aes-ecb-vt-192-46',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000400000000000000000000',
                       ct  = 'FD826CE48E62C5E30867044B86BA4B56')
        AEStestVector( i   = 'aes-ecb-vt-192-47',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000200000000000000000000',
                       ct  = '100E7B831C9F35FA1271F5F1316C6FCF')
        AEStestVector( i   = 'aes-ecb-vt-192-48',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000100000000000000000000',
                       ct  = '0A2DD0C17F68B996AA96C007003D0B31')
        AEStestVector( i   = 'aes-ecb-vt-192-49',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000080000000000000000000',
                       ct  = 'C95F68C57E06B0A2E1F623C83C5D80BF')
        AEStestVector( i   = 'aes-ecb-vt-192-50',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000040000000000000000000',
                       ct  = '571CAFC92C7C8A5EC54C0741E186905C')
        AEStestVector( i   = 'aes-ecb-vt-192-51',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000020000000000000000000',
                       ct  = '22514353E95312C112255E1EED0B2DF6')
        AEStestVector( i   = 'aes-ecb-vt-192-52',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000010000000000000000000',
                       ct  = '791A8BF462BD17580BD9152C6D11C6C5')
        AEStestVector( i   = 'aes-ecb-vt-192-53',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000008000000000000000000',
                       ct  = '5882A0178D548F84A165DB809C60DC28')
        AEStestVector( i   = 'aes-ecb-vt-192-54',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000004000000000000000000',
                       ct  = '3CE4A90EED4458CA6039E42DDADB71C3')
        AEStestVector( i   = 'aes-ecb-vt-192-55',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000002000000000000000000',
                       ct  = 'D3CBAB261207A16BE2751E77044FD7C9')
        AEStestVector( i   = 'aes-ecb-vt-192-56',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000001000000000000000000',
                       ct  = '24E32B698A7B32217093628B01F424AB')
        AEStestVector( i   = 'aes-ecb-vt-192-57',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000800000000000000000',
                       ct  = '9F6AFC0AF27CF565110C77E3C24F4F5B')
        AEStestVector( i   = 'aes-ecb-vt-192-58',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000400000000000000000',
                       ct  = 'E088AA5CDA20EF267BB039B00C72C45B')
        AEStestVector( i   = 'aes-ecb-vt-192-59',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000200000000000000000',
                       ct  = '5CF1018B7E0BA1775601C2E279900360')
        AEStestVector( i   = 'aes-ecb-vt-192-60',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000100000000000000000',
                       ct  = '3B1A7388B89FB9416AD8753CF5AF35D2')
        AEStestVector( i   = 'aes-ecb-vt-192-61',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000080000000000000000',
                       ct  = '137FA4ED00AFCD9F5D8BC0D14BD5837A')
        AEStestVector( i   = 'aes-ecb-vt-192-62',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000040000000000000000',
                       ct  = '806F5C9B663559BB56F234881E4A3E60')
        AEStestVector( i   = 'aes-ecb-vt-192-63',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000020000000000000000',
                       ct  = '8069A449152292DF2DE8642992C632B6')
        AEStestVector( i   = 'aes-ecb-vt-192-64',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000010000000000000000',
                       ct  = '37C6CF2A1ABD1B1F1922B46C7B4A280D')
        AEStestVector( i   = 'aes-ecb-vt-192-65',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000008000000000000000',
                       ct  = '7A2835260E5A0AA2B5DC301800EC8438')
        AEStestVector( i   = 'aes-ecb-vt-192-66',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000004000000000000000',
                       ct  = 'EE81FAF2F9058213FFCACF281CB8509E')
        AEStestVector( i   = 'aes-ecb-vt-192-67',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000002000000000000000',
                       ct  = '57F22D93C37129BA331FDBA38E005A1E')
        AEStestVector( i   = 'aes-ecb-vt-192-68',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000001000000000000000',
                       ct  = 'EC798782E87B7D9F780CC3C3A46519B5')
        AEStestVector( i   = 'aes-ecb-vt-192-69',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000800000000000000',
                       ct  = '43EA28497F5D40E3A4744FA2EDAA42DE')
        AEStestVector( i   = 'aes-ecb-vt-192-70',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000400000000000000',
                       ct  = '91F004E7DEBF41B3414DD8C5C317372C')
        AEStestVector( i   = 'aes-ecb-vt-192-71',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000200000000000000',
                       ct  = 'C249EAE54E7B4DF43B938C1B4CC28314')
        AEStestVector( i   = 'aes-ecb-vt-192-72',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000100000000000000',
                       ct  = '32C289D7EEFB99D2F17AD7B7D45FE1EC')
        AEStestVector( i   = 'aes-ecb-vt-192-73',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000080000000000000',
                       ct  = 'A675FB2E8DDBF810CEF01CF2B728CD2B')
        AEStestVector( i   = 'aes-ecb-vt-192-74',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000040000000000000',
                       ct  = 'A418AAAB6E6921CC731AA8A349386080')
        AEStestVector( i   = 'aes-ecb-vt-192-75',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000020000000000000',
                       ct  = '2E2B0F44863E67D9B0215C4ABD60417F')
        AEStestVector( i   = 'aes-ecb-vt-192-76',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000010000000000000',
                       ct  = 'F0AF7CB19E911D481F6426DAEFDD2240')
        AEStestVector( i   = 'aes-ecb-vt-192-77',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000008000000000000',
                       ct  = 'CB1304DAAA2DF6878F56AC2E0F887E04')
        AEStestVector( i   = 'aes-ecb-vt-192-78',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000004000000000000',
                       ct  = 'B1B70A7E6A0CD1916D9B78BEA19084AE')
        AEStestVector( i   = 'aes-ecb-vt-192-79',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000002000000000000',
                       ct  = '0CDE9F9BE646A5FCE3436B794A9CFC65')
        AEStestVector( i   = 'aes-ecb-vt-192-80',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000001000000000000',
                       ct  = '68C7946D476A0A36674B36AFD7E5DF33')
        AEStestVector( i   = 'aes-ecb-vt-192-81',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000800000000000',
                       ct  = '48770159A07DD8DFFF06C80105F8D57C')
        AEStestVector( i   = 'aes-ecb-vt-192-82',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000400000000000',
                       ct  = '665E62801B3260E3C45BD3BE34DFDEBE')
        AEStestVector( i   = 'aes-ecb-vt-192-83',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000200000000000',
                       ct  = '4159C1F686BFBE5B0E50BDB0DA532B69')
        AEStestVector( i   = 'aes-ecb-vt-192-84',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000100000000000',
                       ct  = '6333100A5A4AD917DC2D4E78A04869A3')
        AEStestVector( i   = 'aes-ecb-vt-192-85',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000080000000000',
                       ct  = '866A4519AB1D199F25886B89D0539ACC')
        AEStestVector( i   = 'aes-ecb-vt-192-86',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000040000000000',
                       ct  = 'EC0CFD37E4CBC7E8BE385283F7AEA75A')
        AEStestVector( i   = 'aes-ecb-vt-192-87',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000020000000000',
                       ct  = 'CA2F383AACCA0810AA13F3E710621422')
        AEStestVector( i   = 'aes-ecb-vt-192-88',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000010000000000',
                       ct  = '1D0EEF6870444F950937831EC0A55D98')
        AEStestVector( i   = 'aes-ecb-vt-192-89',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000008000000000',
                       ct  = '37839B35ED6801E7670496D479A95017')
        AEStestVector( i   = 'aes-ecb-vt-192-90',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000004000000000',
                       ct  = '02317C8C7098C4F94AB867AC7A49DD8D')
        AEStestVector( i   = 'aes-ecb-vt-192-91',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000002000000000',
                       ct  = 'FFB4CB4E3F7F8BF3367EBD43236518B4')
        AEStestVector( i   = 'aes-ecb-vt-192-92',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000001000000000',
                       ct  = '36BEDEF1E4AA3E4A40A305741713FCBF')
        AEStestVector( i   = 'aes-ecb-vt-192-93',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000800000000',
                       ct  = 'B2DFE3C4870269C1E3FEEC39161540D9')
        AEStestVector( i   = 'aes-ecb-vt-192-94',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000400000000',
                       ct  = '147EF2518AD45DA0026056ECBF6A3DFA')
        AEStestVector( i   = 'aes-ecb-vt-192-95',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000200000000',
                       ct  = '027A75E4DE635790E47ACE90D7928804')
        AEStestVector( i   = 'aes-ecb-vt-192-96',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000100000000',
                       ct  = 'C4CF3CCB59BF87D0AFBD629F48CFBB7B')
        AEStestVector( i   = 'aes-ecb-vt-192-97',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000080000000',
                       ct  = '35165C93F564C97E1C32EF97E8151A87')
        AEStestVector( i   = 'aes-ecb-vt-192-98',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000040000000',
                       ct  = '449DE37F7D5A1BBD628ABBE7E061701D')
        AEStestVector( i   = 'aes-ecb-vt-192-99',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000020000000',
                       ct  = 'B1D45EAF218F1799B149BAD677FE129F')
        AEStestVector( i   = 'aes-ecb-vt-192-100',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000010000000',
                       ct  = 'BE08AC6DB6BD0583AA9D2ABC71C73DCD')
        AEStestVector( i   = 'aes-ecb-vt-192-101',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000008000000',
                       ct  = 'BCC835BD3DF1A79E4C7C145B899A5C25')
        AEStestVector( i   = 'aes-ecb-vt-192-102',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000004000000',
                       ct  = '3D311EA611FF5AF371301C58A8E9912D')
        AEStestVector( i   = 'aes-ecb-vt-192-103',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000002000000',
                       ct  = 'A5A1BEA594ACC7CA80F09EA5ADDB5C71')
        AEStestVector( i   = 'aes-ecb-vt-192-104',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000001000000',
                       ct  = '0F09492429FE7222D6CD8190D9F2FFBF')
        AEStestVector( i   = 'aes-ecb-vt-192-105',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000800000',
                       ct  = '816D2220A16B8AAEE71364FD43636C6F')
        AEStestVector( i   = 'aes-ecb-vt-192-106',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000400000',
                       ct  = 'D7E8702408419ED73191B107EAF75A0B')
        AEStestVector( i   = 'aes-ecb-vt-192-107',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000200000',
                       ct  = '9B170EFB1E235B433C78E276BEA082F0')
        AEStestVector( i   = 'aes-ecb-vt-192-108',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000100000',
                       ct  = '03BBECC5598AE974430F29395522F096')
        AEStestVector( i   = 'aes-ecb-vt-192-109',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000080000',
                       ct  = 'DB53517766C0E8CF42059607CBA89380')
        AEStestVector( i   = 'aes-ecb-vt-192-110',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000040000',
                       ct  = '2E2AF4B7931F0AEFFAC5471148A5BB97')
        AEStestVector( i   = 'aes-ecb-vt-192-111',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000020000',
                       ct  = 'C872C0408266403B984F635FF5683DE4')
        AEStestVector( i   = 'aes-ecb-vt-192-112',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000010000',
                       ct  = '15DCF750B0E3A68AD1F4EFD07E8967B4')
        AEStestVector( i   = 'aes-ecb-vt-192-113',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000008000',
                       ct  = 'B41092048E9E6A749F6FD8CE515A23A3')
        AEStestVector( i   = 'aes-ecb-vt-192-114',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000004000',
                       ct  = '4DA9267D62507994312BD5C99ADDE730')
        AEStestVector( i   = 'aes-ecb-vt-192-115',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000002000',
                       ct  = '9E2FCA6D1D626E9C6A924EBF7DBF618A')
        AEStestVector( i   = 'aes-ecb-vt-192-116',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000001000',
                       ct  = 'E092E8D7EF2C2465AEFB2493C3063590')
        AEStestVector( i   = 'aes-ecb-vt-192-117',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000800',
                       ct  = '1C0E58DA37D1068378A88DBE2EDE4E10')
        AEStestVector( i   = 'aes-ecb-vt-192-118',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000400',
                       ct  = '19063F854232B8509A6A3A6D46809959')
        AEStestVector( i   = 'aes-ecb-vt-192-119',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000200',
                       ct  = '447FB09E54EFA285F7530F25C4EA0022')
        AEStestVector( i   = 'aes-ecb-vt-192-120',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000100',
                       ct  = 'F6ABE86321BE40E1FBFDAFED37CC1D9B')
        AEStestVector( i   = 'aes-ecb-vt-192-121',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000080',
                       ct  = '4E8506CD006666341D6CF51F98B41F35')
        AEStestVector( i   = 'aes-ecb-vt-192-122',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000040',
                       ct  = '53995DE0009CA18BECAFB8307C54C14C')
        AEStestVector( i   = 'aes-ecb-vt-192-123',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000020',
                       ct  = '2006BF99F4C58B6CC2627856593FAEEA')
        AEStestVector( i   = 'aes-ecb-vt-192-124',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000010',
                       ct  = '2DA697D2737CB30B744A4644FA1CBC6E')
        AEStestVector( i   = 'aes-ecb-vt-192-125',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000008',
                       ct  = '47A22ACDB60C3A986A8F76ECD0EA3433')
        AEStestVector( i   = 'aes-ecb-vt-192-126',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000004',
                       ct  = 'FDAA17C2CDE20268FE36E164EA532151')
        AEStestVector( i   = 'aes-ecb-vt-192-127',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000002',
                       ct  = '98E7247C07F0FE411C267E4384B0F600')
        AEStestVector( i   = 'aes-ecb-vt-192-128',
                       key = '000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000001',
                       ct  = 'CD33B28AC773F74BA00ED1F312572435')
        AEStestVector( i   = 'aes-ecb-vt-256-1',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '80000000000000000000000000000000',
                       ct  = 'DDC6BF790C15760D8D9AEB6F9A75FD4E')
        AEStestVector( i   = 'aes-ecb-vt-256-2',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '40000000000000000000000000000000',
                       ct  = 'C7098C217C334D0C9BDF37EA13B0822C')
        AEStestVector( i   = 'aes-ecb-vt-256-3',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '20000000000000000000000000000000',
                       ct  = '60F0FB0D4C56A8D4EEFEC5264204042D')
        AEStestVector( i   = 'aes-ecb-vt-256-4',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '10000000000000000000000000000000',
                       ct  = '73376FBBF654D0686E0E84001477106B')
        AEStestVector( i   = 'aes-ecb-vt-256-5',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '08000000000000000000000000000000',
                       ct  = '2F443B52BA5F0C6EA0602C7C4FD259B6')
        AEStestVector( i   = 'aes-ecb-vt-256-6',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '04000000000000000000000000000000',
                       ct  = '75D11B0E3A68C4223D88DBF017977DD7')
        AEStestVector( i   = 'aes-ecb-vt-256-7',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '02000000000000000000000000000000',
                       ct  = '779B38D15BFFB63D8D609D551A5CC98E')
        AEStestVector( i   = 'aes-ecb-vt-256-8',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '01000000000000000000000000000000',
                       ct  = '5275F3D86B4FB8684593133EBFA53CD3')
        AEStestVector( i   = 'aes-ecb-vt-256-9',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00800000000000000000000000000000',
                       ct  = '1CEF2074B336CEC62F12DEA2F6AB1481')
        AEStestVector( i   = 'aes-ecb-vt-256-10',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00400000000000000000000000000000',
                       ct  = '1AEF5ABBAD9D7160874578DCD8BAE172')
        AEStestVector( i   = 'aes-ecb-vt-256-11',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00200000000000000000000000000000',
                       ct  = '46C525DB17E72F26BF03216846B6F609')
        AEStestVector( i   = 'aes-ecb-vt-256-12',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00100000000000000000000000000000',
                       ct  = 'E24411F941BBE08788781E3EC52CBAA4')
        AEStestVector( i   = 'aes-ecb-vt-256-13',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00080000000000000000000000000000',
                       ct  = '83A3DEDD1DD27018F6A6477E40527581')
        AEStestVector( i   = 'aes-ecb-vt-256-14',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00040000000000000000000000000000',
                       ct  = 'B68F8A2CDBAB0C923C67FC8F0F1087DE')
        AEStestVector( i   = 'aes-ecb-vt-256-15',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00020000000000000000000000000000',
                       ct  = '649944A70C32BF87A7409E7AE128FDE8')
        AEStestVector( i   = 'aes-ecb-vt-256-16',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00010000000000000000000000000000',
                       ct  = '2846526D67387539C89314DE9E0C2D02')
        AEStestVector( i   = 'aes-ecb-vt-256-17',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00008000000000000000000000000000',
                       ct  = 'A9A0B8402E53C70DD1688054BA58DDFD')
        AEStestVector( i   = 'aes-ecb-vt-256-18',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00004000000000000000000000000000',
                       ct  = '4A72E6E1B79C83AC4BE3EBA5699EED48')
        AEStestVector( i   = 'aes-ecb-vt-256-19',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00002000000000000000000000000000',
                       ct  = 'B0E36B867BA4FF2B77D0614B0E364E4C')
        AEStestVector( i   = 'aes-ecb-vt-256-20',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00001000000000000000000000000000',
                       ct  = '49B57DE141F6418E3090F24DDD4014B6')
        AEStestVector( i   = 'aes-ecb-vt-256-21',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000800000000000000000000000000',
                       ct  = 'A6C0D5B9797258E1987AC5F6CD20146D')
        AEStestVector( i   = 'aes-ecb-vt-256-22',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000400000000000000000000000000',
                       ct  = '426CF4BDCAA369175965D26E7C71EEA2')
        AEStestVector( i   = 'aes-ecb-vt-256-23',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000200000000000000000000000000',
                       ct  = 'E27F484CE54BC99BC1A52BDA3B518A26')
        AEStestVector( i   = 'aes-ecb-vt-256-24',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000100000000000000000000000000',
                       ct  = 'D16D186284C7E6EE64B8104E0EF20BA5')
        AEStestVector( i   = 'aes-ecb-vt-256-25',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000080000000000000000000000000',
                       ct  = '6431F8538AD54E1E044A9F71F8EF556B')
        AEStestVector( i   = 'aes-ecb-vt-256-26',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000040000000000000000000000000',
                       ct  = 'ECD57CEB451D27EB96C55B2042257E8E')
        AEStestVector( i   = 'aes-ecb-vt-256-27',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000020000000000000000000000000',
                       ct  = '4F0F188DC911B1954AFBC734C9F68872')
        AEStestVector( i   = 'aes-ecb-vt-256-28',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000010000000000000000000000000',
                       ct  = 'B54DEF0337626B65614E81EDFDE620F3')
        AEStestVector( i   = 'aes-ecb-vt-256-29',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000008000000000000000000000000',
                       ct  = '6655D8074CAE0B90B0D3A3FE72D4D9DB')
        AEStestVector( i   = 'aes-ecb-vt-256-30',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000004000000000000000000000000',
                       ct  = 'C6B74B6B9EB4FC0C9A237DB1B616D09A')
        AEStestVector( i   = 'aes-ecb-vt-256-31',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000002000000000000000000000000',
                       ct  = 'D7B5D076EA56EC2B20791D7AD51CCF8F')
        AEStestVector( i   = 'aes-ecb-vt-256-32',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000001000000000000000000000000',
                       ct  = 'FE160C224BF003CE3BDDC90CB52ED22C')
        AEStestVector( i   = 'aes-ecb-vt-256-33',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000800000000000000000000000',
                       ct  = '5E00DA9BA94B5EC0D258D8A8002E0F6A')
        AEStestVector( i   = 'aes-ecb-vt-256-34',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000400000000000000000000000',
                       ct  = '09AC6DCFF4DACFF1651E2BA212A292A3')
        AEStestVector( i   = 'aes-ecb-vt-256-35',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000200000000000000000000000',
                       ct  = 'B283617E318D99AF83A05D9810BA89F7')
        AEStestVector( i   = 'aes-ecb-vt-256-36',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000100000000000000000000000',
                       ct  = '0B5F70CCB40B0EF2538AE9B4A9770B35')
        AEStestVector( i   = 'aes-ecb-vt-256-37',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000080000000000000000000000',
                       ct  = '43282BF180248FB517839B37F4DDAAE4')
        AEStestVector( i   = 'aes-ecb-vt-256-38',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000040000000000000000000000',
                       ct  = 'DDBD534C8B2E6D30A268F88C55AD765B')
        AEStestVector( i   = 'aes-ecb-vt-256-39',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000020000000000000000000000',
                       ct  = 'A41A164E50EC2D9F175E752B755E0B5C')
        AEStestVector( i   = 'aes-ecb-vt-256-40',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000010000000000000000000000',
                       ct  = '37BFF99FF2F7AA97779E4ADF6F13FB10')
        AEStestVector( i   = 'aes-ecb-vt-256-41',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000008000000000000000000000',
                       ct  = '9BA4F7BD298152903A683C4CEC669216')
        AEStestVector( i   = 'aes-ecb-vt-256-42',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000004000000000000000000000',
                       ct  = '5FB750C7CE10DE7B4504248914D0DA06')
        AEStestVector( i   = 'aes-ecb-vt-256-43',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000002000000000000000000000',
                       ct  = '3E748BFA108E086F51D56EC74A9E0FB9')
        AEStestVector( i   = 'aes-ecb-vt-256-44',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000001000000000000000000000',
                       ct  = '31D4E56B99F5B73C1B8437DF332AFB98')
        AEStestVector( i   = 'aes-ecb-vt-256-45',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000800000000000000000000',
                       ct  = '9DC6717B84FC55D266E7B1D9B5C52A5F')
        AEStestVector( i   = 'aes-ecb-vt-256-46',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000400000000000000000000',
                       ct  = '8EF8BA007F23C0A50FC120E07041BCCD')
        AEStestVector( i   = 'aes-ecb-vt-256-47',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000200000000000000000000',
                       ct  = 'C58F38E1839FC1918A12B8C9E88C66B6')
        AEStestVector( i   = 'aes-ecb-vt-256-48',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000100000000000000000000',
                       ct  = 'B695D72A3FCF508C4050E12E40061C2D')
        AEStestVector( i   = 'aes-ecb-vt-256-49',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000080000000000000000000',
                       ct  = '5D2736AD478A50583BC8C11BEFF16D7A')
        AEStestVector( i   = 'aes-ecb-vt-256-50',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000040000000000000000000',
                       ct  = 'DF0EACA8F17847AD41F9578F14C7B56B')
        AEStestVector( i   = 'aes-ecb-vt-256-51',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000020000000000000000000',
                       ct  = 'E5AA14AD48AD0A3C47CC35D5F8020E51')
        AEStestVector( i   = 'aes-ecb-vt-256-52',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000010000000000000000000',
                       ct  = '11BE6C8F58EBD8CEF1A53F591A68E8CE')
        AEStestVector( i   = 'aes-ecb-vt-256-53',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000008000000000000000000',
                       ct  = 'ECFE7BAFCBF42C1FEE015488770B3053')
        AEStestVector( i   = 'aes-ecb-vt-256-54',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000004000000000000000000',
                       ct  = 'E552649F8D8EC4A1E1CD6DF50B6E6777')
        AEStestVector( i   = 'aes-ecb-vt-256-55',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000002000000000000000000',
                       ct  = '521C0629DE93B9119CDB1DDC5809DDEA')
        AEStestVector( i   = 'aes-ecb-vt-256-56',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000001000000000000000000',
                       ct  = 'CB38A62A0BAB1784156BA038CBA99BF6')
        AEStestVector( i   = 'aes-ecb-vt-256-57',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000800000000000000000',
                       ct  = '76CCEE8AAACD394DE1EEF3DDA10CB54B')
        AEStestVector( i   = 'aes-ecb-vt-256-58',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000400000000000000000',
                       ct  = '6AFF910FA1D5673140E2DB59B8416049')
        AEStestVector( i   = 'aes-ecb-vt-256-59',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000200000000000000000',
                       ct  = '064A12C0EF73FB386801BF4F35F3120D')
        AEStestVector( i   = 'aes-ecb-vt-256-60',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000100000000000000000',
                       ct  = '2240E374929D5B1BB8FF0FFDDDF640EC')
        AEStestVector( i   = 'aes-ecb-vt-256-61',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000080000000000000000',
                       ct  = 'D4BA15C904C7692185DE85C02052E180')
        AEStestVector( i   = 'aes-ecb-vt-256-62',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000040000000000000000',
                       ct  = '1714A315AB0166728A44CD91D4AE9018')
        AEStestVector( i   = 'aes-ecb-vt-256-63',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000020000000000000000',
                       ct  = '6C970BDD9F0E222722EA31A1D12DD0AD')
        AEStestVector( i   = 'aes-ecb-vt-256-64',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000010000000000000000',
                       ct  = 'F5956EDF02BD36A401BBB6CE77C3D3FB')
        AEStestVector( i   = 'aes-ecb-vt-256-65',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000008000000000000000',
                       ct  = '0CA11F122CCD7C259DC597EED3DF9BC4')
        AEStestVector( i   = 'aes-ecb-vt-256-66',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000004000000000000000',
                       ct  = '50109AB4912AD2560B206F331B62EB6C')
        AEStestVector( i   = 'aes-ecb-vt-256-67',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000002000000000000000',
                       ct  = 'DBE7C91A4175614889A2D4BEFD64845E')
        AEStestVector( i   = 'aes-ecb-vt-256-68',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000001000000000000000',
                       ct  = '0D3322853A571A6B46B79C0228E0DD25')
        AEStestVector( i   = 'aes-ecb-vt-256-69',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000800000000000000',
                       ct  = '96E4EE0BB9A11C6FB8522F285BADDEB6')
        AEStestVector( i   = 'aes-ecb-vt-256-70',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000400000000000000',
                       ct  = '96705C52D2CFCE82E630C93477C79C49')
        AEStestVector( i   = 'aes-ecb-vt-256-71',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000200000000000000',
                       ct  = 'C50130AED6A126149D71F3888C83C232')
        AEStestVector( i   = 'aes-ecb-vt-256-72',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000100000000000000',
                       ct  = '4816EFE3DEB380566EBA0C17BF582090')
        AEStestVector( i   = 'aes-ecb-vt-256-73',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000080000000000000',
                       ct  = '0390857B4C8C98E4CF7A2B6F3394C507')
        AEStestVector( i   = 'aes-ecb-vt-256-74',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000040000000000000',
                       ct  = '422E73A02025EBE8B8B5D6E0FA24FCB2')
        AEStestVector( i   = 'aes-ecb-vt-256-75',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000020000000000000',
                       ct  = '3271AA7F4BF1D7C38050A43076D4FF76')
        AEStestVector( i   = 'aes-ecb-vt-256-76',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000010000000000000',
                       ct  = 'D2074946F0D37B8975607BFC2E70234C')
        AEStestVector( i   = 'aes-ecb-vt-256-77',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000008000000000000',
                       ct  = '1A509194C1270AB92E5A42D3A9F8D98B')
        AEStestVector( i   = 'aes-ecb-vt-256-78',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000004000000000000',
                       ct  = '512438946360CCC4A5C6D73F6EED7130')
        AEStestVector( i   = 'aes-ecb-vt-256-79',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000002000000000000',
                       ct  = '98CFCDEC46EBEA1A286B3004F2746A0D')
        AEStestVector( i   = 'aes-ecb-vt-256-80',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000001000000000000',
                       ct  = 'A1CF369949677A3AF3D58E3EABF2741B')
        AEStestVector( i   = 'aes-ecb-vt-256-81',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000800000000000',
                       ct  = 'D84C2E1A0E4A52166FA8FF6889D1E5E2')
        AEStestVector( i   = 'aes-ecb-vt-256-82',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000400000000000',
                       ct  = '4AD91CCEEF60119B5078FD162D2735DE')
        AEStestVector( i   = 'aes-ecb-vt-256-83',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000200000000000',
                       ct  = '2860793D818E97AAFF1D339D7702438D')
        AEStestVector( i   = 'aes-ecb-vt-256-84',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000100000000000',
                       ct  = '6F9068BE73364AE250D89D78A6C9CE6F')
        AEStestVector( i   = 'aes-ecb-vt-256-85',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000080000000000',
                       ct  = '024FC3FEF4883FEB1A8DD005305FECCE')
        AEStestVector( i   = 'aes-ecb-vt-256-86',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000040000000000',
                       ct  = '08A61FE0816D75EA15EB3C9FB9CCDED6')
        AEStestVector( i   = 'aes-ecb-vt-256-87',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000020000000000',
                       ct  = '449C86DFA13F260175CE39797686FFA4')
        AEStestVector( i   = 'aes-ecb-vt-256-88',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000010000000000',
                       ct  = '4FFFFC29A59858E1133F2BFB1A8A4817')
        AEStestVector( i   = 'aes-ecb-vt-256-89',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000008000000000',
                       ct  = '19425D1F6480B25096561295697DC2B7')
        AEStestVector( i   = 'aes-ecb-vt-256-90',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000004000000000',
                       ct  = '31974727ECDD2C77C3A428FC3A8CB3FC')
        AEStestVector( i   = 'aes-ecb-vt-256-91',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000002000000000',
                       ct  = 'A57CD704B3C95E744D08DF443458F2F5')
        AEStestVector( i   = 'aes-ecb-vt-256-92',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000001000000000',
                       ct  = '486D8C193DB1ED73ACB17990442FC40B')
        AEStestVector( i   = 'aes-ecb-vt-256-93',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000800000000',
                       ct  = '5E4DBF4E83AB3BC055B9FCC7A6B3A763')
        AEStestVector( i   = 'aes-ecb-vt-256-94',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000400000000',
                       ct  = 'ACF2E0A693FBBCBA4D41B861E0D89E37')
        AEStestVector( i   = 'aes-ecb-vt-256-95',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000200000000',
                       ct  = '32A7CB2AE066A51D2B78FC4B4CFCB608')
        AEStestVector( i   = 'aes-ecb-vt-256-96',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000100000000',
                       ct  = '677D494DBB73CAF55C1990158DA12F14')
        AEStestVector( i   = 'aes-ecb-vt-256-97',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000080000000',
                       ct  = '082A0D2367512ADF0D75A151BFBE0A17')
        AEStestVector( i   = 'aes-ecb-vt-256-98',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000040000000',
                       ct  = '5E5BB7337923C482CE8CBA249E6A8C7D')
        AEStestVector( i   = 'aes-ecb-vt-256-99',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000020000000',
                       ct  = 'D3001BA7C7026EE3E5003179530AFCFC')
        AEStestVector( i   = 'aes-ecb-vt-256-100',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000010000000',
                       ct  = '46EC44F8931E629FE8FD8961312EDDE1')
        AEStestVector( i   = 'aes-ecb-vt-256-101',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000008000000',
                       ct  = 'C5F8ECD79C7B30E81D17E32079969310')
        AEStestVector( i   = 'aes-ecb-vt-256-102',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000004000000',
                       ct  = '5B8AD6919E24CAEBCC55401AEE0C9802')
        AEStestVector( i   = 'aes-ecb-vt-256-103',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000002000000',
                       ct  = 'C2302B7E701B5CC7F8B29E3516DBBFA6')
        AEStestVector( i   = 'aes-ecb-vt-256-104',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000001000000',
                       ct  = 'A1D04D6A76F9F7A94D49FAA64A87F244')
        AEStestVector( i   = 'aes-ecb-vt-256-105',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000800000',
                       ct  = '7FB6F92D35B5CB6C631600EDB9E860BA')
        AEStestVector( i   = 'aes-ecb-vt-256-106',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000400000',
                       ct  = 'B2EF7078BCFACE07AEEC3F9B48830EB3')
        AEStestVector( i   = 'aes-ecb-vt-256-107',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000200000',
                       ct  = 'F475A7493D24C7036E53390374C378B3')
        AEStestVector( i   = 'aes-ecb-vt-256-108',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000100000',
                       ct  = 'B36802AC987377A37BD8EADC97C57D60')
        AEStestVector( i   = 'aes-ecb-vt-256-109',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000080000',
                       ct  = 'ADDCD3D19689C4DDC738CE5F69DC9505')
        AEStestVector( i   = 'aes-ecb-vt-256-110',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000040000',
                       ct  = '0DAF8CA22884915403C0F0BB1F4BD74F')
        AEStestVector( i   = 'aes-ecb-vt-256-111',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000020000',
                       ct  = '4AF36BAE2660503B3248E4685059FD05')
        AEStestVector( i   = 'aes-ecb-vt-256-112',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000010000',
                       ct  = '7D5631814DD8E917D97A0D514C743971')
        AEStestVector( i   = 'aes-ecb-vt-256-113',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000008000',
                       ct  = 'BC3352500FC0CBB9DB5B5F6B491C1BE8')
        AEStestVector( i   = 'aes-ecb-vt-256-114',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000004000',
                       ct  = '6A4A30BA87E87AF65C90AEB7AFEDC76B')
        AEStestVector( i   = 'aes-ecb-vt-256-115',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000002000',
                       ct  = '77E6125897668AC8E73E8C79A6FF8336')
        AEStestVector( i   = 'aes-ecb-vt-256-116',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000001000',
                       ct  = '3FA9D39104EBB323C7AAAA248960DD1E')
        AEStestVector( i   = 'aes-ecb-vt-256-117',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000800',
                       ct  = 'FAD75AD76AB10ADC49036B250E229D39')
        AEStestVector( i   = 'aes-ecb-vt-256-118',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000400',
                       ct  = '2FACAA5FE35B228A16AC74088D702EC4')
        AEStestVector( i   = 'aes-ecb-vt-256-119',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000200',
                       ct  = '88B6CBCFDFEF8AD91720A1BB69A1F33E')
        AEStestVector( i   = 'aes-ecb-vt-256-120',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000100',
                       ct  = 'C7E9D250998632D444356242EF04058D')
        AEStestVector( i   = 'aes-ecb-vt-256-121',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000080',
                       ct  = 'B14DAD8D3D9153F46C0D3A1AD63C7A05')
        AEStestVector( i   = 'aes-ecb-vt-256-122',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000040',
                       ct  = '60ABA678A506608D0845966D29B5F790')
        AEStestVector( i   = 'aes-ecb-vt-256-123',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000020',
                       ct  = '482DC43F2388EF25D24144E144BD834E')
        AEStestVector( i   = 'aes-ecb-vt-256-124',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000010',
                       ct  = '1490A05A7CEE43BDE98B56E309DC0126')
        AEStestVector( i   = 'aes-ecb-vt-256-125',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000008',
                       ct  = 'ABFA77CD6E85DA245FB0BDC5E52CFC29')
        AEStestVector( i   = 'aes-ecb-vt-256-126',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000004',
                       ct  = 'DD4AB1284D4AE17B41E85924470C36F7')
        AEStestVector( i   = 'aes-ecb-vt-256-127',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000002',
                       ct  = 'CEA7403D4D606B6E074EC5D3BAF39D18')
        AEStestVector( i   = 'aes-ecb-vt-256-128',
                       key = '0000000000000000000000000000000000000000000000000000000000000000',
                       pt  = '00000000000000000000000000000001',
                       ct  = '530F8AFBC74536B9A963B4F1C4CB738B')

# Make this test module runnable from the command prompt
if __name__ == "__main__":
    unittest.main()



