/*******************************************************************************
* Regexp.h: Regexp class definition
*-------------------------------------------------------------------------------
* (c)1999-2001 VideoLAN
* $Id: regexp.h,v 1.1 2001/10/06 21:23:36 bozo Exp $
*
* Authors: Benoit Steiner <benny@via.ecp.fr>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
*-------------------------------------------------------------------------------
* This class provides a parser for regular expressions.
* It uses the Berry Sethi algorithm to build the final state machine that will
* parse the expressions
*
*******************************************************************************/


#ifndef _REGEXP_H_
#define _REGEXP_H_




//------------------------------------------------------------------------------
// Class C_RegexpMatch
//------------------------------------------------------------------------------
// An instance of this class represents a match found by the C_Regexp class
// It can be used to obtain relevant information about the location of the
// match in the original string
//------------------------------------------------------------------------------
class C_RegexpMatch
{
 public:
  // Construction of the match
  C_RegexpMatch(const C_String& strMatch, unsigned int iStartIndex,
                unsigned int iEndIndex);
  void AddSubExpr(C_String* pstrSubExpr)
  { ASSERT(pstrSubExpr); m_vSubExpr.Add(pstrSubExpr); }
  
  C_RegexpMatch* Clone() const
  { return new C_RegexpMatch(*this); }

  // Returns the string matching the pattern
  const C_String& GetString() const
  { return m_strMatch; }
  // Returns the index within the input where the match began
  unsigned int GetStartIndex() const
  { return m_iStartIndex; }
  // Returns the index within the input where the match ends
  unsigned int GetEndIndex() const
  { return m_iEndIndex; }

  // Parenthesised subexpres-sions of pattern
  unsigned int GetNbSubExpr() const
  { return m_vSubExpr.Size(); }
  const C_String& GetSubExpr(unsigned int iSubExpr) const
  { return m_vSubExpr[iSubExpr-1]; }
  const C_Vector<C_String>& GetSubExpr() const
  { return m_vSubExpr; }

 private:
  C_String m_strMatch;
  unsigned int m_iStartIndex;
  unsigned int m_iEndIndex;

  C_Vector<C_String> m_vSubExpr;
};



//------------------------------------------------------------------------------
// Class C_Regexp
//------------------------------------------------------------------------------
// Provides a C++ interface for compiling and matching regular expressions 
//------------------------------------------------------------------------------
class C_Regexp
{
 public:
  C_Regexp(const C_String& strPattern);
  ~C_Regexp();
  
  // Check if the string given in the constructor is a valid regexp
  bool IsValid() const
  { return m_bIsValid; }

  // Returns true if the input in its entirety matches the regexp
  // As expensive as GetFirstMatch, so use it only if you don't need the match
  bool DoesMatch(const C_String& strInput) const;

  // Returns the number of matches found in the input
  // As expensive as GetAllMatches, so use it only if you don't need the matches
  unsigned int GetNbMatches(const C_String& strInput) const;

  // Returns the first match found in the input
  // If no match is found, returns NULL
  C_RegexpMatch* GetFirstMatch(const C_String& strInput) const;

  // Returns a vector of all matches found in the input
  // This vector can be empty if no match has been found
  C_Vector<C_RegexpMatch> GetAllMatches(const C_String& strInput) const;

 protected:
  inline C_RegexpMatch* GetMatch(const C_String& strInput, int iStart) const;

 private:
  bool m_bIsValid;

#ifdef HAVE_REGEX_H
  regex_t m_sPattern;
#endif
};



#else
#error "Multiple inclusions of regexp.h"
#endif

